/* Copyright (c) 2015 Axel Wachtler, Daniel Thiele
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id$ */
/**
 * @file
 * @brief Board definition for Atmel SAMR21 Xplained Pro
 *
 * http://www.atmel.com/tools/ATSAMR21-XPRO.aspx
 *
 *
**/

#ifndef BOARD_SAMR21XPRO_H
#define BOARD_SAMR21XPRO_H

#if defined(samr21xpro)
# define RADIO_TYPE (RADIO_AT86RF233)
#endif

#ifndef DEFAULT_SPI_RATE
# define DEFAULT_SPI_RATE  (SPI_RATE_1_2)
#endif

/*=== Hardware Components ============================================*/

/*=== TRX pin access macros ==========================================*/

/* generic macros for SAM devices based on Atmel CMSIS
 * could be relocated to common include file for SAM
 */
#define PIN_GROUP(pin) (pin/32)
#define PIN_MASK(pin)  (1UL << (pin % 32))
#define PINASOUTP(pin) do{ PORT->Group[PIN_GROUP(PIN)].OUTSET.reg = PIN_MASK(pin); }while(0)
#define PINASINP(pin)  do{ PORT->Group[PIN_GROUP(PIN)].OUTSET.reg = PIN_MASK(pin); }while(0)
#define PINHIGH(pin)   do{ PORT->Group[PIN_GROUP(PIN)].OUTSET.reg = PIN_MASK(pin); }while(0)
#define PINLOW(pin)    do{ PORT->Group[PIN_GROUP(PIN)].OUTCLR.reg = PIN_MASK(pin); }while(0)
#define PINGET(pin)    (0 != (PORT->Group[PIN_GROUP(PIN)].IN.reg &= PIN_MASK(pin) ))

/* use pin definitions of Atmel CMSIS */

/**< Pin definition RESET pin */
#define PIN_TRX_RESET  PIN_PB15

/**< Pin definition SLP_TR pin */
#define PIN_TRX_SLPTR  PIN_PA20


/**< RESET pin IO initialization */
#define TRX_RESET_INIT() PINASOUTP(PIN_TRX_RESET)
/**< set RESET pin to high level */
#define TRX_RESET_HIGH() PINHIGH(PIN_TRX_RESET)
/**< set RESET pin to low level */
#define TRX_RESET_LOW()  PINLOW(PIN_TRX_RESET)

/** SLP_TR pin IO initialization */
#define TRX_SLPTR_INIT() PINASOUTP(PIN_TRX_SLPTR)
/** set SLP_TR pin to high level */
#define TRX_SLPTR_HIGH() PINHIGH(PIN_TRX_SLPTR)
/**< set SLP_TR pin to low level */
#define TRX_SLPTR_LOW()  PINLOW(PIN_TRX_SLPTR)


/*=== IRQ access macros ==============================================*/


/** init interrupt handling
 *  - rising edge triggers ICP1 (ICES1),
 *  - timer capture is enabled (ICF1)
 */
# define TRX_IRQ_INIT()  do{                                                         \
    GCLK->CLKCTRL.reg = GCLK_CLKCTRL_CLKEN | GCLK_CLKCTRL_GEN_GCLK6 | EIC_GCLK_ID;   \
    while(0 != (GCLK->STATUS.reg & GCLK_STATUS_SYNCBUSY));                           \
                                                                                     \
    PORT->Group[1].PMUX[0/2].reg = PORT_PMUX_PMUXE_A;                                \
    PORT->Group[1].PINCFG[0].reg = PORT_PINCFG_PMUXEN;                               \
    EIC->CONFIG[0].reg = EIC_CONFIG_SENSE0_RISE;                                     \
    EIC->CTRL.reg |= EIC_CTRL_ENABLE;                                                \
    while(0 != (EIC->STATUS.reg & EIC_STATUS_SYNCBUSY));                             \
    NVIC_EnableIRQ(EIC_IRQn);                                                        \
} while(0)

/** disable TRX interrupt */
#define DI_TRX_IRQ() { EIC->INTENCLR.reg = EIC_INTENSET_EXTINT0;}

/** enable TRX interrupt */
#define EI_TRX_IRQ() { EIC->INTENSET.reg = EIC_INTENSET_EXTINT0; }

/** timestamp register for RX_START event
 * FIXME: add and test the enabling of input capture for separate RX_START (AT86RF231/212)
 *        currently we use the timer register.
 */
#define TRX_TSTAMP_REG

/*=== SPI access macros ==============================================*/
#define SPI_INST SERCOM4->SPI  /**< PORT register for SPI port */

#define PINMUX_SPI_MOSI PINMUX_PB30F_SERCOM4_PAD2  /**< PIN mux for MOSI pin */
#define PINMUX_SPI_MISO PINMUX_PC19F_SERCOM4_PAD0  /**< PIN mux for MISO pin */
#define PINMUX_SPI_SCK  PINMUX_PC18F_SERCOM4_PAD3  /**< PIN mux for SCK pin */
/* regular IO, not using Pinmux for Hardware SS */
#define PIN_SELN PIN_PB31     /**< PIN mux for SS pin */

#define SPI_DATA_REG SPIC.DATA  /**< abstraction for SPI data register */


/**
 * @brief inline function for SPI initialization
 */
static inline void SPI_INIT(uint8_t spirate)
{
    PINASOUTP(PIN_SELN);
}

/** set SS line to low level */
#define SPI_SELN_LOW()  do{ /* TODO: atomic */ PINLOW(PIN_SELN); }while(0)
/** set SS line to high level */
#define SPI_SELN_HIGH() do{ /* TODO: atomic */ PINHIGH(PIN_SELN); }while(0)
/** wait until SPI transfer is ready */
#define SPI_WAITFOR()        do {  } while(0)


/*=== LED access macros ==============================================*/

/*
 * YELLOW : PA19 : 0
 * low-active
 */

#define LED_0_PIN PIN_PA19

#define LED_NUMBER (1)
#define LED_ACTIVITY (0)

#define LED_INIT() \
    do {\
    } while(0)


#define LED_SET(x) \
    do {\
    } while(0)

#define LED_CLR(x) \
    do {\
    } while(0)

#define LED_SET_VALUE(x) \
    do {\
    } while(0)

#define LED_GET_VALUE() ()

#define LED_VAL(msk,val) do{}while(0) /**< \todo how to implement this? */

#define LED_TOGGLE(ln) \
    do {\
    } while(0)


/*=== KEY access macros ==============================================*/


#define PIN_SW0 PIN_PA28
# define KEY_INIT()  PINASINP(PIN_SW0)
# define KEY_GET()   PINGET(PIN_SW0)


/*=== Host Interface ================================================*/

/* Virtual COM of EDBG connected to SERCOM0, Pad[0], Pad[1] */
#define HIF_TYPE (HIF_SERCOM0)


/*=== TIMER Interface ===============================================*/
#define HWTMR_PRESCALE  (1)
#define HWTIMER_TICK    ((1.0*HWTMR_PRESCALE)/F_CPU)
#define HWTIMER_TICK_NB (0xFFFFUL)
#define HWTIMER_REG     (TCD0.CNT)
#define TIMER_TICK      (HWTIMER_TICK_NB * HWTIMER_TICK)
#define TIMER_POOL_SIZE (4)
#define TIMER_INIT() \
    do{ \
    }while(0)
#define TIMER_IRQ_vect

#endif /* BOARD_SAMR21XPRO_H */
