/* Copyright (c) 2010 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: cmdif.c,v 1.23 2010/10/09 14:50:37 dath Exp $ */
/**
 * @file
 * @brief Wireless bootloader application, resides in bootloader section
 *
 * @author Daniel Thiele
 *         Dietzsch und Thiele PartG
 *         Bautzner Str. 67 01099 Dresden
 *         www.ib-dt.de daniel.thiele@ib-dt.de
 *
 * @ingroup grpAppWiBo
 */

#include <stdlib.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>

#include <radio.h>
#include <hif.h>

#include "cmdif.h"
#include "../Inc/wibocmd.h"
#include "wibohost.h"
#include "hexparse.h"

/* variable for function cmd_feedhex()
 * placed here (global) to store in SRAM at compile time
 */
static hexrec_t hexrec;

/*
 * \brief Print "OK" to stdout, used to save code space
 */
static void printok()
{
	puts_P(PSTR("OK"));
}

/*
 * \brief Command to execute wibohost_ping() functions
 *
 * @param params Pointer to list of strings containing the arguments
 *               checking for valid number of arguments is done outside this function
 *
 */
static inline uint8_t cmd_ping(char **params)
{
	volatile uint16_t timeout = 0x7FFF;
	uint16_t nodeid;

	nodeid = strtol(params[0], NULL, 16);
	wibohost_ping(nodeid);

	/* now wait for the reply */
	while (!wibohost_pingreplied() && --timeout)
		; /* software timeout */

	if (!timeout) {
		puts_P(PSTR("ERR ping timeout"));
	} else {
		puts_P(PSTR("OK"));
	}

	return 1;
}

/*
 * \brief Command to execute wibohost_finish() functions
 *
 * @param params Pointer to list of strings containing the arguments
 *               checking for valid number of arguments is done outside this function
 *
 */
static inline uint8_t cmd_finish(char **params)
{
	uint16_t nodeid;

	nodeid = strtol(params[0], NULL, 16);
	wibohost_finish(nodeid);
	printok();
	return 1;
}

/*
 * \brief Command to execute wibohost_feed() functions
 *
 * @param params Pointer to list of strings containing the arguments
 *               checking for valid number of arguments is done outside this function
 *
 * Expected parameters
 *  (1) nodeid
 *  (2) line from intel hex-file
 *
 */
static inline uint8_t cmd_feedhex(char **params)
{
	uint16_t nodeid;
	nodeid = strtol(params[0], NULL, 16);
	if (!parsehexline((uint8_t*) params[1], &hexrec)) {
		puts_P(PSTR("ERR parsing hexline"));
	}
	if (hexrec.type == 0x00) { /* only feed data type lines */
		wibohost_feed(nodeid, hexrec.data, hexrec.len);
		printok();
	} else {
		printf_P(PSTR("ERR ignoring rec type %d\n"), hexrec.type);
	}
	return 1;
}

/*
 * \brief Command to execute wibohost_reset() functions
 *
 * @param params Pointer to list of strings containing the arguments
 *               checking for valid number of arguments is done outside this function
 *
 */
static inline uint8_t cmd_reset(char **params)
{
	wibohost_reset(); /* always reset all nodes */
	printok();
	return 1;
}

/*
 * \brief Command to execute wibohost_exit() functions
 *
 * @param params Pointer to list of strings containing the arguments
 *               checking for valid number of arguments is done outside this function
 *
 *
 * Expected parameters
 *  (1) nodeid
 *
 */
static uint8_t cmd_exit(char **params)
{
	uint16_t nodeid;

	nodeid = strtol(params[0], NULL, 16);
	wibohost_exit(nodeid);
	printok();
	return 1;
}

/*
 * \brief Command to execute wibohost_getcrc() functions
 *
 * @param params Pointer to list of strings containing the arguments
 *               checking for valid number of arguments is done outside this function
 *
 */
static inline uint8_t cmd_checkcrc(char **params)
{
	printf_P(PSTR("OK 0x%04X\n"), wibohost_getcrc());
	return 1;
}

/*
 * \brief Command to echo a string
 * This is intended to check the UART connection
 *
 * @param params Pointer to list of strings containing the arguments
 *               checking for valid number of arguments is done outside this function
 *
 */
static inline uint8_t cmd_echo(char **params)
{
	printf_P(PSTR("OK %s\n"), params[0]);
	return 1;
}

/*
 * \brief Command to echo a string
 * This is intended to check the UART connection
 *
 * @param params Pointer to list of strings containing the arguments
 *               checking for valid number of arguments is done outside this function
 *
 */
static inline uint8_t cmd_info(char **params)
{
	node_config_t *cfg = wibohost_getnodecfg();
	printf_P(PSTR("OK panid=0x%04X shortaddr=0x%04X\n"), cfg->pan_id,
			cfg->short_addr);
	return 1;
}

/*
 * \brief Command to jump into bootloader
 * This is used to support an example application where the host does handle the bootloader itself and
 * an example application residing in the application section of the remote node
 *
 * @param params Pointer to list of strings containing the arguments
 *               checking for valid number of arguments is done outside this function
 *
 *
 * Expected parameters
 *  (1) nodeid
 *
 */
static uint8_t cmd_xmpljbootl(char **params)
{
	uint16_t nodeid;

	nodeid = strtol(params[0], NULL, 16);
	wibohost_xmpljbootl(nodeid);
	printok();
	return 1;
}

/* List of commands that are available at the command interface */
const struct {
	const char *name; /*< Name of the command */
	uint8_t (*execfunc)(char**); /*< Function to be called */
	uint8_t nbparams; /*< Expected number of parameters */
	const char *description; /*< Textual description of the command to print help screen */
} commands[] = {
		{ "ping", cmd_ping, 1, "Ping a node" },
		{ "finish", cmd_finish, 1, "Finish a node (force write)" },
		{ "feedhex", cmd_feedhex, 2, "Feed a line of hex file to a node" },
		{ "reset", cmd_reset, 0, "Reset a node" },
		{ "exit", cmd_exit, 1, "Exit node bootloader" },
		{ "crc", cmd_checkcrc, 0, "Get data CRC" },
		{ "echo", cmd_echo, 1, "Echo a string" },
		{ "info", cmd_info, 0, "Information about myself" },
		{ "xmpljbootl", cmd_xmpljbootl, 1, "Example Application, jump into bootloader" } };

uint8_t process_cmdline(char *ln)
{
	char *params[10];
	uint8_t nbparams;
	uint8_t i;

	nbparams = hif_split_args(ln, 10, params);

	for (i = 0; i < sizeof(commands) / sizeof(commands[0]); i++) {
		if (!strcasecmp(params[0], commands[i].name)) {
			if (commands[i].nbparams == (nbparams - 1)) { /* substract the command itself */
				return (commands[i].execfunc(&params[1]));
			} else {
				puts_P(PSTR("ERR Parameter"));
				return 0;
			}
		}
	}
	puts_P(PSTR("ERR Unknown command"));
	return 0;
}

/* EOF */
