/* Copyright (c) 2008 Axel Wachtler
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: board_stb.h,v 1.12 2009/05/18 05:26:26 awachtler Exp $ */
/**
 * @ingroup grpBoard
 * @file
 * @brief This board is describes the Sensor Terminal Board from
 * Dresden Electronic. It is a carrier board for the RCB family
 *
 * The Sensor Terminal Board is a carrier board for the radio
 * controller board family. 
 * The transceiver wiring fits the common RCBs.
 * The wiring of the radio and the ATmega is shown below:
 *
 *
<pre>
     AVR      RF230
     ---      -----
     PB4  -->  SLPTR
     PD6  <--  CLKM
     PD4  <--  IRQ (ICP1)
     PB5  -->  RSTN
     PB0  -->  SS
     PB2  -->  MOSI
     PB3  <--  MISO
     PB1  -->  SCK

    The STBxxx has memory mapped LEDS and Keys.
    KEY: PE5
    LEDS
     - 2 x memory mapped at 0x4000  (Option 1)
     - 3 x IO mapped at PE2:PE4 (Option 2)


    You can select between using the LEDs on the RCB or on the STB.
    In order to use the RCB LEDS, you have to define the macro USE_RCB_LEDS.

    The LEDs on STB have a write-only memory interface, so we can't read back the LED status.
    So we need to have a shadow register, which stores the current LED state.
    Since the board interface consists only of a header file, we use register
    GPIO2 on ATmega1281 for shadowing, because it would be hard to ensure the
    single instantiation of a global variable from a header file, which is used
    in many module files.


Fuses/Locks:
     LF: 0xe2 - 8MHz internal RC Osc.
     HF: 0x11 - without boot loader
     HF: 0x10 - with boot loader
     EF: 0xff
     LOCK: 0xef - protection of boot section

Original Settings w/ rdk231
     LF: 0x61
     HF: 0x91
     EF: 0xfe

Bootloader:
    Start at byte=0x1e000, address=0xf000, size = 4096 instructions/ 8192 bytes

</pre>

@par Build Options

  - stb230  : STB + RCB230 prior V3.2 / V3.3.1 (AT86RF230A)
  - stb230b : STB + RCB230 V3.2 / RCB230 V3.3.1 (AT86RF230B)
  - stb231  : STB * RCB231 V4.0.2 / RCB231ED V4.1.1
  - stb212  : STB + RCB212SMA V5.3.2

 */

/** ID String for this hardware */
#if defined(stb230)
# define BOARD_TYPE (STB_230)
# define BOARD_NAME_STB2XX "stb230"
# define RADIO_TYPE (RADIO_AT86RF230A)
#elif defined(stb230b)
# define BOARD_TYPE (STB_230B)
# define BOARD_NAME_STB2XX "stb230b"
# define RADIO_TYPE (RADIO_AT86RF230B)
#elif defined(stb231)
# define BOARD_TYPE (STB_231)
# define BOARD_NAME_STB2XX "stb231"
# define RADIO_TYPE (RADIO_AT86RF231)
#elif defined(stb212)
# define BOARD_TYPE (STB_212)
# define BOARD_NAME_STB2XX "stb212"
# define RADIO_TYPE (RADIO_AT86RF212)
#endif

#ifndef BOARD_STB2XX_H
#define BOARD_STB2XX_H
#define BOARD_NAME BOARD_NAME_STB2XX

/*=== Compile time parameters ========================================*/
#ifndef MAX_FRAME_SIZE
# define MAX_FRAME_SIZE (127) /**< maximum allowed frame size */
#endif

#ifndef DEFAULT_SPI_RATE
# define DEFAULT_SPI_RATE  (SPI_RATE_1_2)
#endif

/*=== radio interface definition =====================================*/
#if BOARD_TYPE == STB_230 || BOARD_TYPE == STB_230B
# include "base_rdk230.h"
#else
# include "base_rdk2xx.h"
#endif

static inline void xmem_init(void)
{
    XMCRA |= _BV(SRE);
    XMCRB |= _BV(XMBK);
}


/*=== LED access macros ==============================================*/
#if !defined(USE_RCB_LEDS)
/*=== use LEDs on STB (Memory Mapped) ===*/
# define LED_IO_AD     (0x4000)

# define LED_PORT      (*((volatile uint8_t *)(LED_IO_AD)))
# define LED_SHADOW    GPIOR2
# define LED_MASK      (0x03)
# define LED_SHIFT     (0)
# define LEDS_INVERSE  (1)
# define LED_NUMBER    (2)

# define LED_INIT()\
        do{\
            xmem_init(); \
            LED_SHADOW = LED_MASK;\
            LED_PORT = LED_SHADOW;\
        }while(0)

# define LED_SET_VALUE(x) \
        do{\
            LED_SHADOW = (LED_SHADOW & ~LED_MASK) | ((~x<<LED_SHIFT) & LED_MASK);\
            LED_PORT = LED_SHADOW;\
        }while(0)

# define LED_GET_VALUE()\
        ((~LED_SHADOW & LED_MASK) >> LED_SHIFT)

# define LED_SET(ln)\
        do{\
            LED_SHADOW &= ~(_BV(ln+LED_SHIFT) & LED_MASK);\
            LED_PORT = LED_SHADOW;\
        }while(0)

# define LED_CLR(ln)\
        do{\
            LED_SHADOW |= (_BV(ln+LED_SHIFT) & LED_MASK);\
            LED_PORT = LED_SHADOW;\
        }while(0)

# define LED_VAL(msk,val)\
        do{\
            LED_SHADOW &= ~(LED_MASK|(msk<<LED_SHIFT)); \
            LED_SHADOW |= ~(val & (LED_MASK|msk));\
            LED_PORT = LED_SHADOW;\
        }while(0)


# define LED_TOGGLE(ln)\
        do{\
            LED_SHADOW ^= (_BV(ln+LED_SHIFT) & LED_MASK);\
            LED_PORT = LED_SHADOW;\
        }while(0)

#else
/*=== use LEDs on RCB (IO Mapped) ===*/
# define LED_PORT      PORTE
# define LED_DDR       DDRE
# define LED_MASK      (0x1c)
# define LED_SHIFT     (2)
# define LEDS_INVERSE  (1)
# define LED_NUMBER    (3)
#endif
/*=== KEY access macros ==============================================*/
#define KEY_IO_AD    (0x4000)
#define PIN_KEY       (*(volatile uint8_t*)(KEY_IO_AD))
#define xDDR_KEY       DDRE
#define MASK_KEY      (0x1)
#define SHIFT_KEY     (0)
#define INVERSE_KEYS  (0)
#define PULLUP_KEYS   (0)
#define KEY_INIT      xmem_init

/*=== Host Interface ================================================*/
#define HIF_TYPE      (HIF_FT245)
#define HIF_IO_ENABLE XRAM_ENABLE
#define USB_FIFO_AD  0x2200
#define XRAM_ENABLE() \
        do {\
            DDRC = 0xff; /*force addr 0 to avoid glitches on ext. SRAM*/\
            PORTC = 0x00;\
            XMCRA |= (1 << SRE);\
            XMCRB = (1 << XMBK);\
        }while(0)

#define XRAM_DISABLE() \
        do {\
            XMCRA &= ~(1 << SRE);\
        }while(0)

#define HIF_NO_DATA   (0x0100)
#define FT245_DDR    DDRE
#define FT245_PIN    PINE
#define FT245_TXE    _BV(6)
#define FT245_RXF    _BV(7)
#define FT245_INIT() do { \
           FT245_DDR &= ~(FT245_TXE|FT245_RXF);\
        } while(0)
#define TX_IS_READY        (0)
#define TX_IS_BLOCKED      (FT245_TXE)
#define RX_HAS_DATA        (0)
#define RX_HAS_NO_DATA     (FT245_RXF)
#define FT245_RX_DATA()  ((FT245_PIN & FT245_RXF))
#define FT245_TX_DATA()  ((FT245_PIN & FT245_TXE))

/*=== TIMER Interface ===============================================*/
#define HWTMR_PRESCALE  (1)
#define HWTIMER_TICK    ((1.0*HWTMR_PRESCALE)/F_CPU)
#define HWTIMER_TICK_NB (0xFFFFUL+1)
#define HWTIMER_REG     (TCNT1)
#define TIMER_TICK      (HWTIMER_TICK_NB * HWTIMER_TICK)
#define TIMER_POOL_SIZE     (4)

/** Vector for Timer IRQ routine */
#define TIMER_IRQ_vect   TIMER1_OVF_vect

/**
 * Intialization of the hardware timer T1 (16bit)
 *
 *  - CS1[2:0]  = 1 : Prescaler = 1
 *  - WGM1[3:0] = 0 : Mode = 4 : CTC operation
 *
 * Timer is clocked at F_CPU, and @ref TIMER_IRQ_vect is called every
 * 65535 ticks.
 */
# define TIMER_INIT() \
    do{ \
        TCCR1B |= _BV(CS10); \
        TIMSK1 |= _BV(TOIE1); \
    }while(0)


/*=== OSCCAL tuning =================================================*/
#ifndef TUNED_OSCCAL
# define TUNED_OSCCAL (0xbf)  /* default is 0xb1, but @2.9V 0xbf is better */
#endif
#endif /* BOARD_STB_H*/
