---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET.ADB
--  Description : Root of GUI Widget Hierarchy
--
--  Copyright (C) 2000, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with File_Helpers;
with mcc.Common_Dialogs;
with Ada.Characters.Handling;
with state;
with mcc.Gui.Widget.Label;
with mcc.Gui.Container.Window;
with Subwindow_Actions;
with Ada.Exceptions;
with mcc.Gui.Widget.Rectangle;
with gui.Window;
with Font_Actions;

package body gui.Widget is

   -- size of handles that are created if a widget is selected
   -- Should be even
   Handle_Size : constant Integer := 6;

   function Get_String return String_Pointer is
      Word : Word_Type;
      Last : Natural;
   begin
      File_Helpers.Get_String
        (File_Helpers.Token_Index,
         Word,
         Last,
         Expect_Quotation_Marks => False);
      return new String'(Word (1 .. Last));
   end Get_String;

   -----------------------------------------------------
   -- File is already open and widget type read
   -- read name, x, y, width and height
   -- used in child types to read these common
   -- attributes
   -----------------------------------------------------
   procedure Read_Widget (Widget : in out GUI_Widget) is
   begin -- Read_Widget
      File_Helpers.Token_Index := 2;
      Widget.Name              := Get_String;

      Widget.x                 := Natural'Value (File_Helpers.Token (3).all);
      Widget.y                 := Natural'Value (File_Helpers.Token (4).all);
      Widget.Width             := Natural'Value (File_Helpers.Token (5).all);
      Widget.Height            := Natural'Value (File_Helpers.Token (6).all);
      File_Helpers.Token_Index := 7;

      if File_Helpers.Token_Index > File_Helpers.N_Tokens then
         return;
      end if;
      -- we use an '!' to indicate there is font information
      if File_Helpers.Token (7).all = "!" then
         begin
            Widget.Font_Family       :=
               mcc.Gui.Fonts.Font_Family'Value (File_Helpers.Token (8).all);
            Widget.Font_Size         :=
               Positive'Value (File_Helpers.Token (9).all);
            Widget.Font_Style        :=
               mcc.Gui.Fonts.Font_Style'Value (File_Helpers.Token (10).all);
            Widget.Have_Font         := True;
            File_Helpers.Token_Index := 11;
         exception
            when E : others =>
               Widget.Have_Font := False;
               mcc.Common_Dialogs.Ok_Box
                 (Ada.Exceptions.Exception_Information (E));
         end;
      else
         Widget.Have_Font := False;
      end if;
   end Read_Widget;

   -----------------------------------------------------
   -- write out widget type, name, x, y, width and
   -- height to already created file.
   -- used in child types to write these common
   -- attributes
   -----------------------------------------------------
   procedure Write_Widget
     (Widget : in GUI_Widget)
   is
      use type mcc.Gui.Fonts.Font_Family;
      use File_Helpers;
   begin -- Write_Widget
      Put (Widget.Name.all);
      Put (Natural'Image (Widget.x));
      Put (Natural'Image (Widget.y));
      Put (Natural'Image (Widget.Width));
      Put (Natural'Image (Widget.Height));
      if Widget.Have_Font then
         Put (" ! " & mcc.Gui.Fonts.Font_Family'Image (Widget.Font_Family));
         Put (Natural'Image (Widget.Font_Size));
         Put (' ' & mcc.Gui.Fonts.Font_Style'Image (Widget.Font_Style));
      end if;
   end Write_Widget;

   -----------------------------------------------------
   -- this is not abstract, but null, and should be overridden for
   -- each widget containing action(s).
   -----------------------------------------------------
   procedure Generate_Action_Context_Clause
     (Widget : in GUI_Widget)
   is
   begin -- Generate_Context_Clause
      null;
   end Generate_Action_Context_Clause;

   -- SAG 05.Sep.2000
   procedure Generate_Callback_Action
     (Widget : in GUI_Widget)
   is
   begin
      null;
   end Generate_Callback_Action;

   -- SAG 02.Dec.1999
   procedure Generate_Callback_Context_Clause
     (Widget     : in GUI_Widget;
      Windowname : in String)
   is
   begin
      null;
   end Generate_Callback_Context_Clause;

   Start_X, Start_Y : Integer;
   procedure Widget_Mouse_Listener
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
      use mcc.Gui;
   begin
      if Event.Button = Right and Event.Action = Press then
         Subwindow_Actions.Modify_Widget (mcc.Gui.Widget.Widget'Class (Obj));
      elsif Event.Button = Left and Event.Action = Press then
         mcc.Gui.Set_Cursor
           (Obj    => mcc.Gui.Sized_Object (Obj),
            Cursor => mcc.Gui.Move_Cursor);
         Subwindow_Actions.Select_Widget (mcc.Gui.Widget.Widget'Class (Obj));
         Start_X := Event.X;
         Start_Y := Event.Y;
      elsif Event.Button = Left and Event.Action = Move then
         Subwindow_Actions.Move_Widget
           (mcc.Gui.Widget.Widget'Class (Obj),
            Start_X,
            Start_Y,
            Event.X,
            Event.Y);
      elsif Event.Button = Left and Event.Action = Release then
         mcc.Gui.Set_Cursor
           (Obj    => mcc.Gui.Sized_Object (Obj),
            Cursor => mcc.Gui.Default_Cursor);
      end if;
   end Widget_Mouse_Listener;

   procedure Resize_Mouse_Listener_All
     (Handle : in String;
      Obj    : in out mcc.Gui.Sized_Object'Class;
      Event  : in mcc.Gui.Mouse_Event)
   is
      use mcc.Gui;
   begin
      if Event.Button = Left and Event.Action = Press then
         Start_X := Event.X;
         Start_Y := Event.Y;
      elsif Event.Button = Left and Event.Action = Move then
         Subwindow_Actions.Resize_Widget
           (Handle,
            Start_X,
            Start_Y,
            Event.X,
            Event.Y);
      end if;
   end Resize_Mouse_Listener_All;
   procedure Resize_Mouse_Listener_NW
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
   begin
      Resize_Mouse_Listener_All ("nw", Obj, Event);
   end Resize_Mouse_Listener_NW;
   procedure Resize_Mouse_Listener_N
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
   begin
      Resize_Mouse_Listener_All ("n", Obj, Event);
   end Resize_Mouse_Listener_N;
   procedure Resize_Mouse_Listener_NE
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
   begin
      Resize_Mouse_Listener_All ("ne", Obj, Event);
   end Resize_Mouse_Listener_NE;
   procedure Resize_Mouse_Listener_E
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
   begin
      Resize_Mouse_Listener_All ("e", Obj, Event);
   end Resize_Mouse_Listener_E;
   procedure Resize_Mouse_Listener_SE
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
   begin
      Resize_Mouse_Listener_All ("se", Obj, Event);
   end Resize_Mouse_Listener_SE;
   procedure Resize_Mouse_Listener_S
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
   begin
      Resize_Mouse_Listener_All ("s", Obj, Event);
   end Resize_Mouse_Listener_S;
   procedure Resize_Mouse_Listener_SW
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
   begin
      Resize_Mouse_Listener_All ("sw", Obj, Event);
   end Resize_Mouse_Listener_SW;
   procedure Resize_Mouse_Listener_W
     (Obj   : in out mcc.Gui.Sized_Object'Class;
      Event : in mcc.Gui.Mouse_Event)
   is
   begin
      Resize_Mouse_Listener_All ("w", Obj, Event);
   end Resize_Mouse_Listener_W;

   -----------------------------------------------------
   -- display the widget to a window
   --
   -- place the widget at its proper location
   -- (will be called after it is created by a
   --  method of the same name in a child type)
   -- Also binds the right mouse button to modify_widget
   --            the left mouse button to select_widget
   --            moving left button to moving widget
   -- Cursor is changed while left mouse is depressed on
   -- widget
   -----------------------------------------------------
   procedure Display_Widget
     (Widget    : in out GUI_Widget;
      Container : in out mcc.Gui.Container.Container'Class)
   is
      The_Font : mcc.Gui.Fonts.Font;
   begin
      mcc.Gui.Set_Mouse_Listener
        (Obj      => mcc.Gui.Sized_Object_Pointer (Widget.The_Widget),
         Listener => Widget_Mouse_Listener'Access);

      if Widget.Have_Font then
         mcc.Gui.Fonts.Create
           (Obj    => The_Font,
            Family => Widget.Font_Family,
            Size   => Widget.Font_Size,
            Style  => Widget.Font_Style);
         mcc.Gui.Fonts.Set_Font
           (Obj      => mcc.Gui.Sized_Object (Widget.The_Widget.all),
            New_Font => The_Font);
      end if;

   exception
      when others =>
         mcc.Common_Dialogs.Ok_Box
           ("Unable to display: " & Widget.Name.all & ASCII.LF);
         Undisplay_Widget (Widget);
   end Display_Widget;

   -----------------------------------------------------
   -- highlight a selected widget
   --
   -- calls a create_handles (Tcl) procedure created
   -- in the package begin/end block below
   -- then binds dragging on each of the 8 resize handles
   -- (nw,ne,sw,se,nn,ss,ee,ww)
   -----------------------------------------------------
   procedure Highlight (Widget : in out GUI_Widget) is
      Window : gui.Window.Window_Pointer := state.Get_Current_Window;
   begin
      mcc.Gui.Widget.Rectangle.Create
        (Obj    => Widget.NW_Handle,
         Parent => Window.Display_Area,
         X      => Widget.x - Handle_Size,
         Y      => Widget.y - Handle_Size,
         Width  => Handle_Size,
         Height => Handle_Size);
      mcc.Gui.Widget.Rectangle.Create
        (Obj    => Widget.NE_Handle,
         Parent => Window.Display_Area,
         X      => Widget.x + Widget.Width,
         Y      => Widget.y - Handle_Size,
         Width  => Handle_Size,
         Height => Handle_Size);
      mcc.Gui.Widget.Rectangle.Create
        (Obj    => Widget.N_Handle,
         Parent => Window.Display_Area,
         X      => Widget.x + Widget.Width / 2 - Handle_Size / 2,
         Y      => Widget.y - Handle_Size,
         Width  => Handle_Size,
         Height => Handle_Size);
      mcc.Gui.Widget.Rectangle.Create
        (Obj    => Widget.SW_Handle,
         Parent => Window.Display_Area,
         X      => Widget.x - Handle_Size,
         Y      => Widget.y + Widget.Height,
         Width  => Handle_Size,
         Height => Handle_Size);
      mcc.Gui.Widget.Rectangle.Create
        (Obj    => Widget.SE_Handle,
         Parent => Window.Display_Area,
         X      => Widget.x + Widget.Width,
         Y      => Widget.y + Widget.Height,
         Width  => Handle_Size,
         Height => Handle_Size);
      mcc.Gui.Widget.Rectangle.Create
        (Obj    => Widget.S_Handle,
         Parent => Window.Display_Area,
         X      => Widget.x + Widget.Width / 2 - Handle_Size / 2,
         Y      => Widget.y + Widget.Height,
         Width  => Handle_Size,
         Height => Handle_Size);
      mcc.Gui.Widget.Rectangle.Create
        (Obj    => Widget.W_Handle,
         Parent => Window.Display_Area,
         X      => Widget.x - Handle_Size,
         Y      => Widget.y + Widget.Height / 2 - Handle_Size / 2,
         Width  => Handle_Size,
         Height => Handle_Size);
      mcc.Gui.Widget.Rectangle.Create
        (Obj    => Widget.E_Handle,
         Parent => Window.Display_Area,
         X      => Widget.x + Widget.Width,
         Y      => Widget.y + Widget.Height / 2 - Handle_Size / 2,
         Width  => Handle_Size,
         Height => Handle_Size);
      mcc.Gui.Set_Mouse_Listener
        (Obj      => Widget.NW_Handle'Unchecked_Access,
         Listener => Resize_Mouse_Listener_NW'Access);
      mcc.Gui.Set_Mouse_Listener
        (Obj      => Widget.N_Handle'Unchecked_Access,
         Listener => Resize_Mouse_Listener_N'Access);
      mcc.Gui.Set_Mouse_Listener
        (Obj      => Widget.NE_Handle'Unchecked_Access,
         Listener => Resize_Mouse_Listener_NE'Access);
      mcc.Gui.Set_Mouse_Listener
        (Obj      => Widget.E_Handle'Unchecked_Access,
         Listener => Resize_Mouse_Listener_E'Access);
      mcc.Gui.Set_Mouse_Listener
        (Obj      => Widget.SE_Handle'Unchecked_Access,
         Listener => Resize_Mouse_Listener_SE'Access);
      mcc.Gui.Set_Mouse_Listener
        (Obj      => Widget.S_Handle'Unchecked_Access,
         Listener => Resize_Mouse_Listener_S'Access);
      mcc.Gui.Set_Mouse_Listener
        (Obj      => Widget.SW_Handle'Unchecked_Access,
         Listener => Resize_Mouse_Listener_SW'Access);
      mcc.Gui.Set_Mouse_Listener
        (Obj      => Widget.W_Handle'Unchecked_Access,
         Listener => Resize_Mouse_Listener_W'Access);
      mcc.Gui.Set_Cursor
        (Obj    => mcc.Gui.Sized_Object (Widget.NW_Handle),
         Cursor => mcc.Gui.Resize_NW);
      mcc.Gui.Set_Cursor
        (Obj    => mcc.Gui.Sized_Object (Widget.N_Handle),
         Cursor => mcc.Gui.Resize_N);
      mcc.Gui.Set_Cursor
        (Obj    => mcc.Gui.Sized_Object (Widget.NE_Handle),
         Cursor => mcc.Gui.Resize_NE);
      mcc.Gui.Set_Cursor
        (Obj    => mcc.Gui.Sized_Object (Widget.E_Handle),
         Cursor => mcc.Gui.Resize_E);
      mcc.Gui.Set_Cursor
        (Obj    => mcc.Gui.Sized_Object (Widget.SE_Handle),
         Cursor => mcc.Gui.Resize_SE);
      mcc.Gui.Set_Cursor
        (Obj    => mcc.Gui.Sized_Object (Widget.S_Handle),
         Cursor => mcc.Gui.Resize_S);
      mcc.Gui.Set_Cursor
        (Obj    => mcc.Gui.Sized_Object (Widget.SW_Handle),
         Cursor => mcc.Gui.Resize_SW);
      mcc.Gui.Set_Cursor
        (Obj    => mcc.Gui.Sized_Object (Widget.W_Handle),
         Cursor => mcc.Gui.Resize_W);

   exception
      when e : others =>
         mcc.Common_Dialogs.Ok_Box
           ("Highlight: " & Ada.Exceptions.Exception_Information (e));
   end Highlight;

   -----------------------------------------------------
   -- highlight a selected widget
   --
   -- done by calling Tcl procedure move_handles
   -- defined below in the package begin/end block
   -----------------------------------------------------
   procedure Move_Highlight (Widget : in out GUI_Widget) is
   begin
      mcc.Gui.Widget.Rectangle.Move
        (Obj => Widget.NW_Handle,
         X   => Widget.x - Handle_Size,
         Y   => Widget.y - Handle_Size);
      mcc.Gui.Widget.Rectangle.Move
        (Obj => Widget.NE_Handle,
         X   => Widget.x + Widget.Width,
         Y   => Widget.y - Handle_Size);
      mcc.Gui.Widget.Rectangle.Move
        (Obj => Widget.N_Handle,
         X   => Widget.x + Widget.Width / 2 - Handle_Size / 2,
         Y   => Widget.y - Handle_Size);
      mcc.Gui.Widget.Rectangle.Move
        (Obj => Widget.SW_Handle,
         X   => Widget.x - Handle_Size,
         Y   => Widget.y + Widget.Height);
      mcc.Gui.Widget.Rectangle.Move
        (Obj => Widget.SE_Handle,
         X   => Widget.x + Widget.Width,
         Y   => Widget.y + Widget.Height);
      mcc.Gui.Widget.Rectangle.Move
        (Obj => Widget.S_Handle,
         X   => Widget.x + Widget.Width / 2 - Handle_Size / 2,
         Y   => Widget.y + Widget.Height);
      mcc.Gui.Widget.Rectangle.Move
        (Obj => Widget.W_Handle,
         X   => Widget.x - Handle_Size,
         Y   => Widget.y + Widget.Height / 2 - Handle_Size / 2);
      mcc.Gui.Widget.Rectangle.Move
        (Obj => Widget.E_Handle,
         X   => Widget.x + Widget.Width,
         Y   => Widget.y + Widget.Height / 2 - Handle_Size / 2);
   exception
      when others =>
         null;
   end Move_Highlight;

   -----------------------------------------------------
   -- calls the destroy_handles (Tcl) procedure
   -- defined below in package begin/end block
   -----------------------------------------------------
   procedure Unhighlight (Widget : in GUI_Widget) is
   begin
      mcc.Gui.Widget.Rectangle.Destroy (Widget.NW_Handle);
      mcc.Gui.Widget.Rectangle.Destroy (Widget.N_Handle);
      mcc.Gui.Widget.Rectangle.Destroy (Widget.NE_Handle);
      mcc.Gui.Widget.Rectangle.Destroy (Widget.E_Handle);
      mcc.Gui.Widget.Rectangle.Destroy (Widget.SE_Handle);
      mcc.Gui.Widget.Rectangle.Destroy (Widget.S_Handle);
      mcc.Gui.Widget.Rectangle.Destroy (Widget.SW_Handle);
      mcc.Gui.Widget.Rectangle.Destroy (Widget.W_Handle);
   exception
      when others =>
         null;
   end Unhighlight;

   -----------------------------------------------------
   -- Tell Tcl to destroy the widget!
   -----------------------------------------------------
   procedure Undisplay_Widget (Widget : in GUI_Widget) is
   begin
      mcc.Gui.Destroy (mcc.Gui.Object (Widget.The_Widget.all));
   exception
      when others =>
         null;
   end Undisplay_Widget;

   -----------------------------------------------------
   -- Algorithm:
   -- 1) Get dialog name
   -- 2) Change dialog appearance to transient
   -- 3) If widget has name (won't if just created),
   --    set entry1
   -- 4) set entries2-5 to x,y,width,height
   -- 5) Bind clicking window close to
   --    delete_properties_action (see tcl_bindings.txt)
   --    this calls Subwindow_Actions.Cancel_Properties_Dialog
   -- 6) Put cursor in entry1
   -- Note: other properties set in method of same name
   -- in child class
   -----------------------------------------------------
   procedure Set_Properties (Widget : in out GUI_Widget) is
      use type mcc.Gui.Widget.Label.Label_Pointer;
   begin

      if Widget.Name /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Name_Entry.all,
            Text => Widget.Name.all);
      end if;

      mcc.Gui.Widget.Text_Entry.Set_Text
        (Obj  => Widget.X_Entry.all,
         Text => Widget.x);

      mcc.Gui.Widget.Text_Entry.Set_Text
        (Obj  => Widget.Y_Entry.all,
         Text => Widget.y);

      mcc.Gui.Widget.Text_Entry.Set_Text
        (Obj  => Widget.Width_Entry.all,
         Text => Widget.Width);

      mcc.Gui.Widget.Text_Entry.Set_Text
        (Obj  => Widget.Height_Entry.all,
         Text => Widget.Height);

      mcc.Gui.Container.Window.Set_Close_Handler
        (Obj     => Widget.Properties,
         Handler => Subwindow_Actions.Cancel_Properties_Dialog'Access);

      mcc.Gui.Widget.Text_Entry.Highlight
        (Obj   => Widget.Name_Entry.all,
         Start => 0,
         Stop  => 0);

      if Widget.Font_Label /= null then
         if Widget.Have_Font then
            mcc.Gui.Widget.Label.Set_Text
              (Obj  => Widget.Font_Label.all,
               Text => "Font: " &
                       Font_Actions.Display_Enum
                          (mcc.Gui.Fonts.Font_Family'Image
                              (Widget.Font_Family)));
            mcc.Gui.Widget.Label.Set_Text
              (Obj  => Widget.Style_Label.all,
               Text =>
                 Font_Actions.Display_Enum
                    (mcc.Gui.Fonts.Font_Style'Image (Widget.Font_Style)) &
                 mcc.Gui.Fonts.Font_Size'Image (Widget.Font_Size) &
                 " pt");
         else
            mcc.Gui.Widget.Label.Set_Text
              (Obj  => Widget.Font_Label.all,
               Text => "Font:");
            mcc.Gui.Widget.Label.Set_Text
              (Obj  => Widget.Style_Label.all,
               Text => "Default");
         end if;
      end if;
   end Set_Properties;

   -----------------------------------------------------
   -- Algorithm:
   -- 1) Get dialog name
   -- 2) Read Widget_Name into Word
   -- 3) Read x,y,width,height directly into window
   --    3a) set to -99 if invalid text
   -- Note: other properties applied in method of same
   -- name in child class
   -- Widget name can't begin upper case-- fix this!
   -----------------------------------------------------
   procedure Apply_Properties (Widget : in out GUI_Widget) is
   begin
      declare
         Name : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Name_Entry.all);
      begin
         Widget.Name := new String'(Name);
      end;

      begin
         Widget.x := mcc.Gui.Widget.Text_Entry.Get_Text (Widget.X_Entry.all);
      exception
         when others =>
            Widget.x := -99;
      end;

      begin
         Widget.y := mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Y_Entry.all);
      exception
         when others =>
            Widget.y := -99;
      end;

      begin
         Widget.Width :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Width_Entry.all);
      exception
         when others =>
            Widget.Width := -99;
      end;

      begin
         Widget.Height :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Height_Entry.all);
      exception
         when others =>
            Widget.Height := -99;
      end;

   end Apply_Properties;

   -----------------------------------------------------
   -- Check properties common to all widgets.  Others
   -- checked from method of same name in child class
   --
   -- Make sure name isn't null
   -- Make sure name is alphanumeric and has no "_"
   -- If not OK, highlight entry1
   --
   -- Note we stop checking as soon as we find something
   -- that isn't OK.
   --
   -- Check x,y,width,height to make sure they are positive
   -- and highlight if not
   -----------------------------------------------------
   procedure Check_Properties
     (Widget : in out GUI_Widget;
      Ok     : out Boolean)
   is
   begin
      Ok := Widget.Name /= null and then Widget.Name.all'Length >= 1;

      if Ok then
         for i in Widget.Name.all'Range loop
            if not Ada.Characters.Handling.Is_Alphanumeric
                     (Widget.Name.all (i))
              and then not (Widget.Name.all (i) = '_')
            then
               Ok := False;
            end if;
         end loop;
      end if;

      if not Ok then
         mcc.Gui.Widget.Text_Entry.Highlight (Widget.Name_Entry.all);
      end if;

      if Ok and then Widget.x < 0 then
         Ok := False;
         mcc.Gui.Widget.Text_Entry.Highlight (Widget.X_Entry.all);
      end if;

      if Ok and then Widget.y < 0 then
         Ok := False;
         mcc.Gui.Widget.Text_Entry.Highlight (Widget.Y_Entry.all);
      end if;

      if Ok and then Widget.Width < 0 then
         Ok := False;
         mcc.Gui.Widget.Text_Entry.Highlight (Widget.Width_Entry.all);
      end if;

      if Ok and then Widget.Height < 0 then
         Ok := False;
         mcc.Gui.Widget.Text_Entry.Highlight (Widget.Height_Entry.all);
      end if;
   end Check_Properties;

   -----------------------------------------------------
   -- Destroy the dialog.  Note that we don't have one
   -- running anymore. (Can only have 1 at a time since
   -- the dialog names are reused).
   -----------------------------------------------------
   procedure Close_Properties (Widget : in GUI_Widget) is
   begin
      mcc.Gui.Destroy (mcc.Gui.Object (Widget.Properties.all));
      state.Set_Dialog_Running (False);
   end Close_Properties;

   -----------------------------------------------------
   -- return the name
   -----------------------------------------------------
   function Get_Name (Widget : in Widget_Access) return String is
   begin
      return Widget.Name.all;
   end Get_Name;

   -------------------------------------------------
   -- function Get_The_Widget
   -------------------------------------------------
   function Get_The_Widget
     (Widget : in Widget_Access)
      return   mcc.Gui.Widget.Widget_Pointer
   is
   begin
      return Widget.The_Widget;
   end Get_The_Widget;

   -------------------------------------------------
   -- procedure Set_Location
   --
   -- set x,y,width and height for a widget
   -------------------------------------------------
   procedure Set_Location
     (Widget              : in Widget_Access;
      x, y, Width, Height : in Integer)
   is
   begin
      Widget.x      := x;
      Widget.y      := y;
      Widget.Width  := Width;
      Widget.Height := Height;
   end Set_Location;

   procedure Generate_Widget_Font
     (Widget : in GUI_Widget)
   is
      procedure P (S : String) is
      begin
         File_Helpers.P (S);
      end P;
   begin
      if not Widget.Have_Font then
         return;
      end if;
      File_Helpers.Set_Indent (2);
      P ("declare");
      P ("   use Mcc.Gui.Fonts;");
      P ("   The_Font : Mcc.Gui.Fonts.Font;");
      P ("begin");
      P ("   Mcc.Gui.Fonts.Create");
      P ("     (Obj    => The_Font,");
      P ("      Family => " &
         mcc.Gui.Fonts.Font_Family'Image (Widget.Font_Family) & ",");
      P ("      Size   => " &
         mcc.Gui.Fonts.Font_Size'Image (Widget.Font_Size) & ",");
      P ("      Style  => " &
         mcc.Gui.Fonts.Font_Style'Image (Widget.Font_Style) & ");");
      P ("   Mcc.Gui.Fonts.Set_Font");
      P ("     (Obj      => Mcc.Gui.Sized_Object(" & Widget.Name.all & "),");
      P ("      New_Font => The_Font);");
      P ("end;");
   end Generate_Widget_Font;

   function Has_Anything_To_Fill_Or_Read
     (Widget : in GUI_Widget)
      return   Boolean
   is
   begin
      return False;
   end Has_Anything_To_Fill_Or_Read;

   procedure Generate_Fill_Action (Widget : in GUI_Widget) is
   begin
      null;
   end Generate_Fill_Action;

   procedure Generate_Read_Action (Widget : in GUI_Widget) is
   begin
      null;
   end Generate_Read_Action;

end Gui.Widget;
