---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-RADIO_BUTTON.ADB
--  Description : GUI Widget Radio Button
--
--  Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Ada.Text_IO;
with File_Helpers;
with Gui_Enum;
with radio_button_dialog_window;
with mcc.Gui.Widget.Text_Entry;   use type mcc.Gui.Widget.Widget_Pointer;
with mcc.Gui.Widget.Button.Radio;
with mcc.Common_Dialogs;
with Group_Manager;
with Generate_Helpers;

package body gui.Widget.Radio_Button is

   -- reads information from file into GUI_Widget,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Radio_Button) is
      Word : Word_Type;
      Last : Natural;
   begin
      Read_Widget (GUI_Widget (Widget));
      Widget.Text              := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      Widget.Group             := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
   end Read_Widget;

   -- Writes information to file from GUI_Widget

   procedure Write_Widget (Widget : in Radio_Button) is
   begin
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.RadioButton) & " ");
      Write_Widget (GUI_Widget (Widget));
      File_Helpers.Put (" """);
      File_Helpers.Put_String (Widget.Text.all);
      File_Helpers.Put ("""");
      File_Helpers.P (" """ & Widget.Group.all & """", Indent => False);
   end Write_Widget;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Radio_Button) is
   begin
      Generate_Helpers.Generate_With ("Mcc.Gui.Widget.Button.Radio");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Radio_Button) is
   begin
      File_Helpers.P
        (Widget.Name.all &
         " : aliased Mcc.Gui.Widget.Button.Radio.Radio_Button;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Radio_Button;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      P ("Mcc.Gui.Widget.Button.Radio.Create");
      P ("  (Obj    => " & Widget.Name.all & ",");
      P ("   Parent => " & Window_Name & ",");
      P ("   X      =>" & Integer'Image (Widget.x) & ",");
      P ("   Y      =>" & Integer'Image (Widget.y) & ",");
      P ("   Width  =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height =>" & Integer'Image (Widget.Height) & ",");
      P ("   Text   => """ &
         Generate_Helpers.Quote_String (Widget.Text.all) & """);");
      P ("Mcc.Gui.Widget.Button.Radio.Add_To_Group");
      P ("  (Group    => " & Widget.Group.all & ",");
      P ("   Button   => " & Widget.Name.all & "'access);");
   end Generate_Widget_Creation;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Radio_Button;
      Container : in out mcc.Gui.Container.Container'Class)
   is
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.Gui.Widget.Button.Radio.Radio_Button;
      end if;

      mcc.Gui.Widget.Button.Radio.Create
        (Obj    =>
           mcc.Gui.Widget.Button.Radio.Radio_Button (Widget.The_Widget.all),
         Parent => Container,
         X      => Widget.x,
         Y      => Widget.y,
         Width  => Widget.Width,
         Height => Widget.Height,
         Text   => Widget.Text.all);
      Group_Manager.Add_To_Group
        (Obj   =>
           mcc.Gui.Widget.Button.Radio.Radio_Button (Widget.The_Widget.all),
         Group => Widget.Group);

      gui.Widget.Display_Widget
        (Widget    => gui.Widget.GUI_Widget (Widget),
         Container => Container);

   exception
      when others =>
         mcc.Common_Dialogs.Ok_Box ("Can't display: " & Widget.Name.all);
   end Display_Widget;

   procedure Set_Properties (Widget : in out Radio_Button) is
   begin
      radio_button_dialog_window.Generate_Window;
      Widget.Properties   :=
        radio_button_dialog_window.radio_button_dialog_window'Access;
      Widget.Name_Entry   := radio_button_dialog_window.entry1'Access;
      Widget.X_Entry      := radio_button_dialog_window.entry2'Access;
      Widget.Y_Entry      := radio_button_dialog_window.entry3'Access;
      Widget.Width_Entry  := radio_button_dialog_window.entry4'Access;
      Widget.Height_Entry := radio_button_dialog_window.entry5'Access;
      Widget.Text_Entry   := radio_button_dialog_window.entry6'Access;
      Widget.Group_Entry  := radio_button_dialog_window.entry7'Access;

      Widget.Font_Label  := radio_button_dialog_window.font_label'Access;
      Widget.Style_Label := radio_button_dialog_window.font_style'Access;

      Set_Properties (GUI_Widget (Widget));
      if Widget.Text /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Text_Entry.all,
            Text => Widget.Text.all);
      end if;
      if Widget.Group /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Group_Entry.all,
            Text => Widget.Group.all);
      end if;
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Radio_Button) is
   begin
      Apply_Properties (GUI_Widget (Widget));
      declare
         Text : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Text_Entry.all);
      begin
         Widget.Text := new String'(Text);
      end;
      declare
         Group : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Group_Entry.all);
      begin
         Widget.Group := new String'(Group);
      end;
   end Apply_Properties;

   procedure Check_Properties
     (Widget : in out Radio_Button;
      Ok     : out Boolean)
   is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);
      if Ok then
         Ok := (Widget.Text /= null) and then (Widget.Text.all /= "");
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.Text_Entry.all);
         end if;
      end if;

      if Ok then
         Ok := (Widget.Group /= null) and then (Widget.Group.all /= "");
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.Group_Entry.all);
         end if;
      end if;

      if not Ok then
         mcc.Gui.Bell;
      end if;

   end Check_Properties;

end Gui.Widget.Radio_Button;
