---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  File Name : gui-widget-frame.adb
--
--  Author    : SA Griffin
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
--
-- Copyright (C) 2005, SA Griffin <stephen.griffin.external@eads.com>
---------------------------------------------------------------------

with Ada.Text_IO;
with Ada.Exceptions;
with Ada.Strings.Fixed;

with Gui_Enum;
with File_Helpers;
with Generate_Helpers;
with frame_dialog_window;
with mcc.Common_Dialogs;
with mcc.Gui.Widget.Button.Check;
with mcc.Gui.Widget.Frame;        use type mcc.Gui.Widget.Widget_Pointer;

package body gui.Widget.Frame is

   -- reads information from file into Frame,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Frame) is
   begin
      Read_Widget (GUI_Widget (Widget));

      Widget.Border_Width      :=
         Natural'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

   exception
      when E : others =>
         Ada.Text_IO.New_Line;
         Ada.Text_IO.Put (Ada.Exceptions.Exception_Information (E));
   end Read_Widget;

   -- Writes information to file fromFrame
   procedure Write_Widget (Widget : in Frame) is
   begin
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Frame) & " ");

      Write_Widget (GUI_Widget (Widget));

      File_Helpers.P
        (Natural'Image (Widget.Border_Width), Indent => False);

   exception
      when E : others =>
         Ada.Text_IO.New_Line;
         Ada.Text_IO.Put (Ada.Exceptions.Exception_Information (E));
   end Write_Widget;

   procedure Generate_Widget_Context_Clause (Widget : in Frame) is
   begin
      null;
   end Generate_Widget_Context_Clause;

   procedure Generate_Widget_Declaration (Widget : in Frame) is
   begin
      null;
   end Generate_Widget_Declaration;

   procedure Generate_Widget_Creation
     (Widget      : in Frame;
      Window_Name : in String)
   is
      S_File : Ada.Text_IO.File_Type;
      B_File : Ada.Text_IO.File_Type;

      procedure PS (Text : String := "") is
      begin
         Ada.Text_IO.Put_Line (S_File, Text);
      end PS;

      procedure PB (Text : String := "") is
      begin
         Ada.Text_IO.Put_Line (B_File, Text);
      end PB;

      Filename : constant String :=
         File_Helpers.Convert_Window_Name (Widget.Name.all);
   begin

      --
      -- Add the Call to Create Frame
      --
      File_Helpers.P;
      File_Helpers.P (Widget.Name.all & ".Create;");

      --
      -- Create the Package Spec and Body for the Frame Create and Clear
      -- function.
      --
      Ada.Text_IO.Create
        (File => S_File,
         Name => Filename & ".ads",
         Mode => Ada.Text_IO.Out_File);

      PS ("--");
      PS ("-- Package spec: " & Widget.Name.all);
      PS ("--");
      PS ("-- Automatically generated by RAPID");
      PS ("--");
      PS ("with Mcc.Gui.Container.Frame;");
      PS;
      PS ("package " & Widget.Name.all & " is");
      PS;
      PS ("   Frame : aliased Mcc.Gui.Container.Frame.Frame;");
      PS;
      PS ("   procedure Create;");
      PS;
      PS ("   procedure Clear;");
      PS;
      PS ("end " & Widget.Name.all & ";");

      Ada.Text_IO.Close (File => S_File);

      Ada.Text_IO.Create
        (File => B_File,
         Name => Filename & ".adb",
         Mode => Ada.Text_IO.Out_File);

      PB ("--");
      PB ("-- Package body: " & Widget.Name.all);
      PB ("--");
      PB ("-- Automatically generated by RAPID");
      PB ("--");
      PB ("with Ada.Text_IO;");
      PB ("with Ada.Exceptions;");
      PB ("with Mcc.Gui.Widget.Label;");
      PB ("with " & Window_Name & ";");
      PB;
      PB ("package body " & Widget.Name.all & " is");
      PB;
      PB ("   Label_Lab : aliased Mcc.Gui.Widget.Label.Label;");
      PB;
      PB ("   procedure Create is");
      PB ("   begin");
      PB;
      PB ("      Mcc.Gui.Container.Frame.Create");
      PB ("        (Obj          => Frame, ");
      PB ("         Parent       => " & Window_Name & "." & Window_Name & ",");
      PB ("         X            =>" & Integer'Image (Widget.x) & ",");
      PB ("         Y            =>" & Integer'Image (Widget.y) & ",");
      PB ("         Width        =>" & Integer'Image (Widget.Width) & ",");
      PB ("         Height       =>" & Integer'Image (Widget.Height) & ",");
      PB ("         Border_Width =>" &
         Integer'Image (Widget.Border_Width) & ");");
      PB;
      PB ("   exception");
      PB ("      when E : others =>");
      PB ("         Ada.Text_IO.New_Line;");
      PB ("         Ada.Text_IO.Put (Ada.Exceptions.Exception_Information (E));");
      PB ("   end Create;");
      PB;
      PB ("   procedure Clear is");
      PB ("   begin");
      PB;
      PB ("      Mcc.Gui.Widget.Label.Create");
      PB ("        (Obj     => label_lab,");
      PB ("         Parent  => Frame,");
      PB ("         X       => 1,");
      PB ("         Y       => 1,");
      PB ("         Width   =>" & Integer'Image (Widget.Width - 3) & ",");
      PB ("         Height  =>" & Integer'Image (Widget.Height - 3) & ",");
      PB ("         Justify => Mcc.Gui.Widget.Label.Center,");
      PB ("         Text    => " & '"' & ' ' & '"' & ");");
      PB;
      PB ("   exception");
      PB ("      when E:others =>");
      PB ("         Ada.Text_IO.New_Line;");
      PB ("         Ada.Text_IO.Put (Ada.Exceptions.Exception_Information (E));");
      PB ("   end Clear;");
      PB;
      PB ("end " & Widget.Name.all & ";");

      Ada.Text_IO.Close (File => B_File);

   exception

      when E : others =>
         Ada.Text_IO.New_Line;
         Ada.Text_IO.Put (Ada.Exceptions.Exception_Information (E));

   end Generate_Widget_Creation;

   procedure Generate_Action_Context_Clause (Widget : in Frame) is
   begin
      Generate_Helpers.Generate_With (Widget.Name.all);
   end Generate_Action_Context_Clause;

   procedure Display_Widget
     (Widget    : in out Frame;
      Container : in out mcc.Gui.Container.Container'Class)
   is

   begin

      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.Gui.Widget.Frame.Frame;
      end if;

      mcc.Gui.Widget.Frame.Create
        (Obj          => mcc.Gui.Widget.Frame.Frame (Widget.The_Widget.all),
         Parent       => Container,
         X            => Widget.x,
         Y            => Widget.y,
         Width        => Widget.Width,
         Height       => Widget.Height,
         Border_Width => Widget.Border_Width);

      Display_Widget (Widget => GUI_Widget (Widget), Container => Container);

   exception
      when e : others =>
         mcc.Common_Dialogs.Ok_Box
           ("Can't display: " &
            Widget.Name.all &
            ASCII.LF &
            Ada.Exceptions.Exception_Information (e));

   end Display_Widget;

   procedure Set_Properties (Widget : in out Frame) is
   begin

      frame_dialog_window.Generate_Window;

      Widget.Properties   := frame_dialog_window.frame_dialog_window'Access;
      Widget.Name_Entry   := frame_dialog_window.entry1'Access;
      Widget.X_Entry      := frame_dialog_window.entry2'Access;
      Widget.Y_Entry      := frame_dialog_window.entry3'Access;
      Widget.Width_Entry  := frame_dialog_window.entry4'Access;
      Widget.Height_Entry := frame_dialog_window.entry5'Access;
      Widget.Border_Width_Entry := frame_dialog_window.Border_TE'Access;

      Set_Properties (GUI_Widget (Widget));

      mcc.Gui.Widget.Text_Entry.Set_Text
        (Obj  => Widget.Border_Width_Entry.all,
         Text => Widget.Border_Width);

   exception

      when E : others =>
         Ada.Text_IO.New_Line;
         Ada.Text_IO.Put (Ada.Exceptions.Exception_Information (E));

   end Set_Properties;

   procedure Apply_Properties (Widget : in out Frame) is
   begin

      Apply_Properties (GUI_Widget (Widget));

      begin

         Widget.Border_Width :=
            mcc.Gui.Widget.Text_Entry.Get_Text
              (Widget.Border_Width_Entry.all);

      exception
         when others =>
            Widget.Border_Width := 1;
      end;

   exception

      when E : others =>
         Ada.Text_IO.New_Line;
         Ada.Text_IO.Put (Ada.Exceptions.Exception_Information (E));

   end Apply_Properties;

   procedure Check_Properties (Widget : in out Frame; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);
      if not Ok then
         mcc.Gui.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Frame;
