---------------------------------------------------------------
--
--  FILE_HELPERS.ADS
--  Description : IO helpers
--
--  Copyright (C) 2000, Martin C. Carlisle <carlislem@acm.org>
--
-- FILE_HELPERS is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- FILE_HELPERS is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Ada.Text_IO;

package File_Helpers is

   ------------------------------------------------------------------------
   -- PROCEDURE Convert_Window_Name
   --
   -- Change window name as follows:
   -- 1) change "." to "main"
   -- 2) replace '.' and ' ' with '_'
   -- 3) remove all trailing '_'
   -- 4) convert letters to lowercase
   ------------------------------------------------------------------------
   function Convert_Window_Name (Window_Name : in String) return String;

   ------------------------------------------------------------------------
   -- Global file descriptor for all input operations
   ------------------------------------------------------------------------
   Input_File : Ada.Text_IO.File_Type;

   ------------------------------------------------------------------------
   -- PROCEDURE Get_Line
   --
   -- Gets the next line from the Input_File and tokenizes the line into
   -- the Token array.
   -- If the line retrieved is empty or consists of only spaces/tabs
   -- then following line(s) are retrieved until a non-empty line is
   -- found.
   -- Sets N_Tokens to the number of tokens tokenized, or sets
   -- N_Tokens to 0 on end of file.
   ------------------------------------------------------------------------
   type String_Access is access String;
   Max_Tokens_Per_Line : constant := 32;
   subtype Token_Range is Positive range 1 .. Max_Tokens_Per_Line;
   Token    : array (Token_Range) of String_Access;
   N_Tokens : Natural range 0 .. Max_Tokens_Per_Line := 0;
   -- Token_Index is a working pointer into the Token array
   -- and is furnished for usage by the application.
   -- The code in File_Helpers does not depend on its setting.
   Token_Index : Natural range 0 .. Max_Tokens_Per_Line := 0;

   procedure Get_Line;

   ------------------------------------------------------------------------
   -- PROCEDURE Get_String
   --
   -- expects the Token at the given Token_Index to contain
   -- a string. If the string begins and ends with " then those
   -- are removed.
   -- As in Ada, embedded quotation marks are written as "".
   -- Doubled quotation marks are reduced to single in the output.
   -- If
   -- * the given Token_Index does not exist (i.e. N_Tokens is
   --   smaller than Token_Index), or
   -- * Token(Token_Index) is the empty string, or
   -- * Expect_Quotation_Marks is set to True and the surrounding
   --   quotation marks are not found
   -- then Last is returned as 0, else Last denotes the position
   -- of the last valid character in the output Item.
   ------------------------------------------------------------------------
   procedure Get_String
     (Token_Index            : in Token_Range;
      Item                   : out String;
      Last                   : out Natural;
      Expect_Quotation_Marks : in Boolean := True);

   ------------------------------------------------------------------------
   -- Global file descriptor for all output operations
   ------------------------------------------------------------------------
   Output_File : Ada.Text_IO.File_Type;

   ------------------------------------------------------------------------
   -- PROCEDURE Put_String
   --
   -- writes string to Output_File enclosed in "".
   -- If string contains '"' then two quotation marks are written.
   ------------------------------------------------------------------------
   procedure Put_String (Item : in String);

   ------------------------------------------------------------------------
   -- PROCEDURE P
   --
   -- writes string to Output_File followed by a newline.
   -- If Indent is True then first writes indentation.
   -- If Newline is True then prints a newline after the item.
   ------------------------------------------------------------------------
   procedure P
     (Item : in String := "";
      Indent : Boolean := True;
      Newline : Boolean := True);

   ------------------------------------------------------------------------
   -- PROCEDURE Put
   --
   -- writes string to Output_File as-is.
   -- Shorthand for P (Item, False, False)
   ------------------------------------------------------------------------
   procedure Put (Item : in String);

   ------------------------------------------------------------------------
   -- PROCEDURE Set_Indent
   --
   -- sets the indentation level which is used by procedures Put and P.
   -- Level 0 means no indentation, 1 means one indentation, etc.
   -- One indentation is 3 spaces.
   ------------------------------------------------------------------------
   procedure Set_Indent (Level : Natural);

end File_Helpers;

