---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-PROGRESS.ADB
--  Description : GUI Widget Progress Bar
--
--  Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Gui_Enum;
with File_Helpers;
with Generate_Helpers;
with mcc.Common_Dialogs;
with progress_dialog_Window;
with mcc.Gui.Widget.Progress; use type mcc.Gui.Widget.Widget_Pointer;
with mcc.Gui.Colors;

package body gui.Widget.Progress is
   -- reads information from file into Progress,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Progress) is
   begin
      Read_Widget (GUI_Widget (Widget));

      Widget.Fg_Color          := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      Widget.Bg_Color          := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
   end Read_Widget;

   -- Writes information to file from Progress
   procedure Write_Widget (Widget : in Progress) is
   begin -- Write_Widget
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Progress) & " ");
      Write_Widget (GUI_Widget (Widget));
      File_Helpers.Put (" " & Widget.Fg_Color.all);
      File_Helpers.P (" " & Widget.Bg_Color.all, Indent => False);
   end Write_Widget;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Progress) is
   begin
      Generate_Helpers.Generate_With ("Mcc.Gui.Widget.Progress");
      Generate_Helpers.Generate_With ("Mcc.Gui.Colors");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Progress) is
   begin
      File_Helpers.P
        (Widget.Name.all & " : aliased Mcc.Gui.Widget.Progress.Progress;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Progress;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      P ("Mcc.Gui.Widget.Progress.Create");
      P ("  (Obj     => " & Widget.Name.all & ",");
      P ("   Parent  => " & Window_Name & ",");
      P ("   X       =>" & Integer'Image (Widget.x) & ",");
      P ("   Y       =>" & Integer'Image (Widget.y) & ",");
      P ("   Width   =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height  =>" & Integer'Image (Widget.Height) & ");");
      if Widget.Bg_Color.all /= "default" then
         P ("Mcc.Gui.Widget.Progress.Set_Background_Color");
         P ("  (Obj       => " & Widget.Name.all & ",");
         P ("   New_Color => Mcc.Gui.Colors.Named_Color");
         P ("     (Mcc.Gui.Colors." & Widget.Bg_Color.all & "));");
      end if;

      if Widget.Fg_Color.all /= "default" then
         P ("Mcc.Gui.Widget.Progress.Set_Foreground_Color");
         P ("  (Obj       => " & Widget.Name.all & ",");
         P ("   New_Color => Mcc.Gui.Colors.Named_Color");
         P ("     (Mcc.Gui.Colors." & Widget.Fg_Color.all & "));");
      end if;
   end Generate_Widget_Creation;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Progress;
      Container : in out mcc.Gui.Container.Container'Class)
   is
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.Gui.Widget.Progress.Progress;
      end if;

      mcc.Gui.Widget.Progress.Create
        (Obj    => mcc.Gui.Widget.Progress.Progress (Widget.The_Widget.all),
         Parent => Container,
         X      => Widget.x,
         Y      => Widget.y,
         Width  => Widget.Width,
         Height => Widget.Height);
      if Widget.Fg_Color.all /= "default" then
         mcc.Gui.Widget.Set_Foreground_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.Gui.Colors.Named_Color (mcc.Gui.Colors.Color_Enum'Value
                                             (Widget.Fg_Color.all)));
      end if;
      if Widget.Bg_Color.all /= "default" then
         mcc.Gui.Widget.Set_Background_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.Gui.Colors.Named_Color (mcc.Gui.Colors.Color_Enum'Value
                                             (Widget.Bg_Color.all)));
      end if;
      mcc.Gui.Widget.Progress.Set_Progress
        (Obj    => mcc.Gui.Widget.Progress.Progress (Widget.The_Widget.all),
         Amount => 0.5);
      Display_Widget (GUI_Widget (Widget), Container);
   exception
      when others =>
         mcc.Common_Dialogs.Ok_Box ("Can't display: " & Widget.Name.all);
   end Display_Widget;

   procedure Set_Properties (Widget : in out Progress) is
   begin
      progress_dialog_Window.Generate_Window;
      Widget.Properties   :=
        progress_dialog_Window.progress_dialog_Window'Access;
      Widget.Name_Entry   := progress_dialog_Window.entry1'Access;
      Widget.X_Entry      := progress_dialog_Window.entry2'Access;
      Widget.Y_Entry      := progress_dialog_Window.entry3'Access;
      Widget.Width_Entry  := progress_dialog_Window.entry4'Access;
      Widget.Height_Entry := progress_dialog_Window.entry5'Access;
      Widget.Fg_Entry     := progress_dialog_Window.entry6'Access;
      Widget.Bg_Entry     := progress_dialog_Window.entry7'Access;
      Set_Properties (GUI_Widget (Widget));
      if Widget.Bg_Color /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Bg_Entry.all,
            Text => Widget.Bg_Color.all);
      end if;
      if Widget.Fg_Color /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Fg_Entry.all,
            Text => Widget.Fg_Color.all);
      end if;
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Progress) is
   begin
      Apply_Properties (GUI_Widget (Widget));

      declare
         FG_Color : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Fg_Entry.all);
      begin
         if FG_Color = "" then
            Widget.Fg_Color := new String'("default");
         else
            Widget.Fg_Color := new String'(FG_Color);
         end if;
      end;
      declare
         BG_Color : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Bg_Entry.all);
      begin
         if BG_Color = "" then
            Widget.Bg_Color := new String'("default");
         else
            Widget.Bg_Color := new String'(BG_Color);
         end if;
      end;
   end Apply_Properties;

   procedure Check_Properties (Widget : in out Progress; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);
      if Ok then
         Ok := (Widget.Fg_Color /= null) and then (Widget.Fg_Color.all /= "");
         if Ok and then Widget.Fg_Color.all /= "default" then
            declare
               Color : mcc.Gui.Colors.Color_Enum;
            begin
               Color := mcc.Gui.Colors.Color_Enum'Value (Widget.Fg_Color.all);
            exception
               when others =>
                  Ok := False;
            end;
         end if;
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.Fg_Entry.all);
         end if;
      end if;
      if Ok then
         Ok := (Widget.Bg_Color /= null) and then (Widget.Bg_Color.all /= "");
         if Ok and then Widget.Bg_Color.all /= "default" then
            declare
               Color : mcc.Gui.Colors.Color_Enum;
            begin
               Color := mcc.Gui.Colors.Color_Enum'Value (Widget.Bg_Color.all);
            exception
               when others =>
                  Ok := False;
            end;
         end if;
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.Bg_Entry.all);
         end if;
      end if;

      if not Ok then
         mcc.Gui.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Progress;
