# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023,2024 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# References:
#    . Ares Galaxy 246 source code
#    . Forensic Analysis of Ares Galaxy Peer-to-Peer Network (Kolenbrander)
#
# Ares Galaxy main files (* decoded by MobiusFT):
#
#  . DHTNodes.dat - DHT nodes
#       @see DHT_readnodeFile - DHT/dhtzones.pas (line 125)
#       (client ID, IP, udp_port, tcp_port, type)
#
#  . MDHTNodes.dat - MDHT nodes
#       @see MDHT_readnodeFile - BitTorrent/dht_zones.pas (line 124)
#       (client ID, IP, udp_port, type)
#
#  . PHashIdx.dat, PhashIdxTemp.dat, TempPHash.dat - PHash table
#       @see ICH_load_phash_indexs - helper_ICH.pas (line 1023)
#       (hash_sha1, Phash table)
#
#  * ShareH.dat - Trusted metadata
#       @see get_trusted_metas - helper_library_db.pas (line 542)
#
#  * ShareL.dat - Cached metadata
#       @see get_cached_metas - helper_library_db.pas (line 367)
#
#  . SNodes.dat
#       @see aresnodes_loadfromdisk - helper_ares_nodes (line 445)
#       (IP, port, reports, attempts, connects, first_seen, last_seen)
#
#  * TorrentH.dat - DHT magnet file history and metadata
#       @see tthread_dht.getMagnetFiles - DHT/thread_dht.pas (line 284)
#
#  . TempDL/PHash_XXX.dat - Downloading file pieces info
#       @see ICH_loadPieces - helper_ICH (line 528)
#       (flag_done, progress, hash_sha1)
#
#  . TempDL/PBTHash_XXX.dat - Downloading file (BitTorrent) metadata
#       @see BitTorrentDb_load - BitTorrent/BitTorrentDlDb.pas (line 88)
#
#  * ___ARESTRA___*.* - Downloading files, with metadata info
#       @see read_details_DB_Download - helper_download_disk.pas (line 722)
#
#  . __INCOMPLETE__*.* - Downloading files (BitTorrent)
#
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
from . import decoder_arestra


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
APP_ID = "ares"
APP_NAME = "Ares Galaxy"


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Ares Profile class
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Profile(object):

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Initialize object
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __init__(self, folder):
        self.__folder = folder
        self.path = folder.path.replace('/', '\\')
        self.app_id = APP_ID
        self.app_name = APP_NAME

        self.__downloads_loaded = False
        self.__downloads = []

        self.__arestra_loaded = False
        self.__arestra = {}

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Get downloaded files
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def get_downloads(self):
        self.__load_downloads()

        return self.__downloads

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Load downloaded files
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __load_downloads(self):
        if self.__downloads_loaded:
            return

        self.__load_arestra()

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Load ___ARESTRA___ files
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __load_arestra(self):
        if self.__arestra_loaded:
            return

        user_profile = self.user_profile

        folder = user_profile.get_entry_by_path('%desktop%/My Shared Folder')
        self.__retrieve_my_shared_folder(folder)

        folder = user_profile.get_entry_by_path('%localappdata%/Ares/My Shared Folder')
        self.__retrieve_my_shared_folder(folder)

        folder = user_profile.get_entry_by_path('%{374de290-123f-4565-9164-39c4925e467b}%')
        self.__retrieve_my_shared_folder(folder)

        self.__arestra_loaded = True

    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    # @brief Load ___ARESTRA___ files from folder
    # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
    def __retrieve_my_shared_folder(self, folder):
        if not folder:
            return

        for child in folder.get_children():
            fname = child.name.lower()

            if child.is_reallocated():
                pass

            elif child.is_folder():
                self.__retrieve_my_shared_folder(child)

            elif fname.startswith('___arestra___'):
                entry = decoder_arestra.decode(child)

                if entry:
                    self.__arestra[entry.hash_sha1] = entry
