// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "ui.h"
#include <mobius/exception.inc>
#include <stdexcept>
#include <string>
#include <unordered_map>

namespace mobius
{
namespace ui
{
namespace
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Implementation info structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
struct implementation
{
  std::string id;
  std::string description;
  impl_builder builder;
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Global UI implementation
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static std::shared_ptr <ui_impl_base> g_impl_;
static std::string g_impl_id_;
static std::unordered_map <std::string, implementation> implementations_;

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Add UI implementation class
//! \param id UI implementation ID
//! \param description UI implementation description
//! \param f UI implementation builder function
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
add_implementation (
  const std::string& id,
  const std::string& description,
  impl_builder f)
{
  implementations_[id] = implementation {id, description, f};
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Remove UI implementation class
//! \param id UI implementation ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
remove_implementation (const std::string& id)
{
  implementations_.erase (id);

  if (g_impl_id_ == id)
    {
      g_impl_.reset (); 
      g_impl_id_.clear ();
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get list of UI implementation classes
//! \return List of implementation classes ID and description
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
impl_list
get_implementations ()
{
  impl_list implementations;
  
  for (auto& i : implementations_)
    {
      implementations.push_back (
        std::make_pair (i.second.id, i.second.description)
      );
    }
  
  return implementations;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set current UI implementation class
//! \param id UI implementation ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
set_implementation (const std::string& id)
{
  if (g_impl_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("UI implementation already set"));

  // search implementation
  auto iter = implementations_.find (id);

  if (iter != implementations_.end ())
    {
      g_impl_ = iter->second.builder ();
      g_impl_id_ = id;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get current UI implementation class
//! \return Smart pointer to UI implementation class
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::shared_ptr <ui_impl_base>
get_implementation ()
{
  // if g_impl_ is not set, create using first UI implementation available
  if (!g_impl_)
    {
      if (implementations_.size () == 0)
        throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("no UI implementation found"));

      else
        {
          g_impl_ = implementations_.begin ()->second.builder ();
          g_impl_id_ = implementations_.begin ()->first;
        }
    }

  // return implementation instance
  return g_impl_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ui::ui ()
  : impl_ (get_implementation ())
{
}

} // namespace ui
} // namespace mobius
