// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \file map.cc C++ API <i>mobius.pod.map</i> class wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <pymobius.h>
#include "map.h"
#include <mobius/exception.inc>
#include <mobius/string_functions.h>
#include <stdexcept>
#include "pyobject.h"
#include "data.h"
#include "api_dataholder.h"

namespace {

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create Python object from POD map
//! \param value POD map
//! \return Python object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
map_to_object (const mobius::pod::map& value)
{
  api_dataholder_o *data = api_dataholder_new ();

  for (const auto& p : value)
    {
      if (p.first != ".object")
        api_dataholder_setattr (data, p.first, pymobius_pod_data_to_pyobject (p.second));
    }

  return reinterpret_cast <PyObject *> (data);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create <i>map</i> from Python object
//! \param obj Python object
//! \return POD map
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static mobius::pod::map
map_from_object (PyObject *obj)
{
  mobius::pod::map map;

  PyObject *py_list = PyObject_Dir (obj);
      
  if (py_list)
    {
      auto size = PySequence_Size (py_list);

      for (Py_ssize_t i = 0; i < size;i++)
        {
          mobius::py::pyobject py_attr_name = PySequence_GetItem (py_list, i);

          if (py_attr_name)
            {
              mobius::py::pyobject py_attr_value = PyObject_GetAttr (obj, py_attr_name);
                  
              auto cpp_key = mobius::py::pystring_as_std_string (py_attr_name);

              if (!mobius::string::startswith (cpp_key, "__") && !py_attr_value.is_callable ())
                {
                  auto cpp_value = pymobius_pod_data_from_pyobject (py_attr_value);
                  map.set (cpp_key, cpp_value);
                }
            }
        }

      Py_DECREF (py_list);
      map.set (".object", mobius::pod::data ());
    }
    
  return map;
}

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Check if object type is <i>map</i>
//! \param pyobj Python object
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
pymobius_pod_map_check (PyObject *pyobj)
{
  return PyObject_IsInstance (pyobj, (PyObject *) &pod_map_t);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create <i>map</i> Python object from C++ object
//! \param map POD map object
//! \return new map object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
pymobius_pod_map_to_pyobject (const mobius::pod::map& map)
{
  PyObject *ret = nullptr;

  if (map.contains (".object"))
    ret = map_to_object (map);

  else
    {
      ret = _PyObject_New (&pod_map_t);

      if (ret)
        ((pod_map_o *) ret)->obj = new mobius::pod::map (map);
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create <i>map</i> C++ object from Python object
//! \param py_value Python object
//! \return map object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::pod::map
pymobius_pod_map_from_pyobject (PyObject *py_value)
{
  mobius::pod::map map;

  if (pymobius_pod_map_check (py_value))
    map = * (reinterpret_cast <pod_map_o *>(py_value)->obj);
  
  else if (PyDict_Check (py_value))
    {
      PyObject *key, *value;
      Py_ssize_t pos = 0;
      
      while (PyDict_Next (py_value, &pos, &key, &value))
        {
          auto cpp_key = mobius::py::pystring_as_std_string (key);
          auto cpp_value = pymobius_pod_data_from_pyobject (value);
          
          map.set (cpp_key, cpp_value);
        }
    }

  // Read attributes from Python object
  else
    map = map_from_object (py_value);

  return map;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get_size</i> method implementation
//! \param self Object
//! \param args Argument list
//! \return Number of items
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_size (pod_map_o *self, PyObject *)
{
  // Execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = mobius::py::pybool_from_bool (self->obj->get_size ());
    }
  catch (const std::exception& e)
    {
      mobius::py::set_runtime_error (e.what ());
    }

  // Return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>contains</i> method implementation
//! \param self Object
//! \param args Argument list
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_contains (pod_map_o *self, PyObject *args)
{
  // Parse input args
  std::string arg_key;

  try
    {
      arg_key = mobius::py::get_arg_as_std_string (args, 0);
    }
  catch (const std::exception& e)
    {
      mobius::py::set_invalid_type_error (e.what ());
      return nullptr;
    }

  // Execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = mobius::py::pybool_from_bool (self->obj->contains (arg_key));
    }
  catch (const std::exception& e)
    {
      mobius::py::set_runtime_error (e.what ());
    }

  // Return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get</i> method implementation
//! \param self Object
//! \param args Argument list
//! \return Data object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get (pod_map_o *self, PyObject *args)
{
  // Parse input args
  std::string arg_key;
  mobius::pod::data arg_varg;

  try
    {
      arg_key = mobius::py::get_arg_as_std_string (args, 0);
      arg_varg = mobius::py::get_arg_as_cpp (
              args, 1, pymobius_pod_data_from_pyobject, mobius::pod::data ()
      );
    }
  catch (const std::exception& e)
    {
      mobius::py::set_invalid_type_error (e.what ());
      return nullptr;
    }

  // Execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = pymobius_pod_data_to_pyobject (self->obj->get (arg_key, arg_varg));
    }
  catch (const std::exception& e)
    {
      mobius::py::set_runtime_error (e.what ());
    }

  // Return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>set</i> method implementation
//! \param self Object
//! \param args Argument list
//! \return None
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_set (pod_map_o *self, PyObject *args)
{
  // Parse input args
  std::string arg_key;
  mobius::pod::data arg_value;

  try
    {
      arg_key = mobius::py::get_arg_as_std_string (args, 0);
      arg_value = mobius::py::get_arg_as_cpp (args, 1, pymobius_pod_data_from_pyobject);
    }
  catch (const std::exception& e)
    {
      mobius::py::set_invalid_type_error (e.what ());
      return nullptr;
    }

  // Execute C++ function
  try
    {
      self->obj->set (arg_key, arg_value);
    }
  catch (const std::exception& e)
    {
      mobius::py::set_runtime_error (e.what ());
    }

  // return None
  return mobius::py::pynone ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>remove</i> method implementation
//! \param self Object
//! \param args Argument list
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_remove (pod_map_o *self, PyObject *args)
{
  // Parse input args
  std::string arg_key;

  try
    {
      arg_key = mobius::py::get_arg_as_std_string (args, 0);
    }
  catch (const std::exception& e)
    {
      mobius::py::set_invalid_type_error (e.what ());
      return nullptr;
    }

  // Execute C++ function
  try
    {
      self->obj->remove (arg_key);
    }
  catch (const std::exception& e)
    {
      mobius::py::set_runtime_error (e.what ());
      return nullptr;
    }

  // return None
  return mobius::py::pynone ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>to_python</i> method implementation
//! \param self Object
//! \param args Argument list
//! \return Data object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_to_python (pod_map_o *self, PyObject *)
{
  PyObject *ret = nullptr;

  try
    {
      ret = mobius::py::pydict_from_cpp_container (
              *self->obj,
              mobius::py::pystring_from_std_string,
              pymobius_pod_data_to_pyobject
            );
    }
  catch (const std::exception& e)
    {
      mobius::py::set_runtime_error (e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get_values</i> method implementation
//! \param self Object
//! \param args Argument list
//! \return Data object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_values (pod_map_o *self, PyObject *)
{
  PyObject *ret = nullptr;

  try
    {
      auto f = [](const auto& p){
        return mobius::py::pytuple_from_cpp_pair (
                p,
                mobius::py::pystring_from_std_string,
                pymobius_pod_data_to_pyobject);
      };

      ret = mobius::py::pylist_from_cpp_container (*self->obj, f);
    }
  catch (const std::exception& e)
    {
      mobius::py::set_runtime_error (e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef tp_methods[] =
{
  {
    (char *) "get_size",
    (PyCFunction) tp_f_get_size,
    METH_VARARGS,
    "Get map size"
  },
  {
    (char *) "contains",
    (PyCFunction) tp_f_contains,
    METH_VARARGS,
    "Check if map contains a given key"
  },
  {
    (char *) "get",
    (PyCFunction) tp_f_get,
    METH_VARARGS,
    "Get item"
  },
  {
    (char *) "set",
    (PyCFunction) tp_f_set,
    METH_VARARGS,
    "Set item"
  },
  {
    (char *) "remove",
    (PyCFunction) tp_f_remove,
    METH_VARARGS,
    "Remove item"
  },
  {
    (char *) "to_python",
    (PyCFunction) tp_f_to_python,
    METH_VARARGS,
    "Convert map to Python dict"
  },
  {
    (char *) "get_values",
    (PyCFunction) tp_f_get_values,
    METH_VARARGS,
    "Get items"
  },
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>map</i> Constructor
//! \param type Type object
//! \param args Argument list
//! \param kwds Keywords dict
//! \return new <i>map</i> object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_new (PyTypeObject *, PyObject *args, PyObject *)
{
  PyObject *ret = nullptr;

  try
    {
      mobius::pod::map map;

      if (mobius::py::get_arg_size (args) > 0)
        map = pymobius_pod_map_from_pyobject (mobius::py::get_arg (args, 0));

      ret = pymobius_pod_map_to_pyobject (map);
    }
  catch (const std::exception& e)
    {
      mobius::py::set_runtime_error (e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>map</i> deallocator
//! \param self Object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
tp_dealloc (pod_map_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject pod_map_t =
{
  PyVarObject_HEAD_INIT (NULL, 0)
  "mobius.pod.map",                        		// tp_name
  sizeof (pod_map_o),                      		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) tp_dealloc,                 		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "map class",                             		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  tp_methods,                              		// tp_methods
  0,                                       		// tp_members
  0,                               		        // tp_getset
  &pod_data_t,                             		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  tp_new,                                       		// tp_new
  0,                                       		// tp_free
  0,                                       		// tp_is_gc
  0,                                       		// tp_bases
  0,                                       		// tp_mro
  0,                                       		// tp_cache
  0,                                       		// tp_subclasses
  0,                                       		// tp_weaklist
  0,                                       		// tp_del
  0,                                       		// tp_version_tag
};
