# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import libxml2
XML_ENCODING='utf-8'

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generic object's instance
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class PyObject (object):
  pass

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Persistence layer
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Pickle (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_prop (self, node, name):
    value = node.prop (name)
    if value:
      value = value.decode (XML_ENCODING)
    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __set_prop (self, node, name, value):
    if value != None:
      node.setProp (name, value.encode (XML_ENCODING))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load XML file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self, path):
    doc = libxml2.parseFile (path)
    node = doc.getRootElement ()
    name, value = self.load_item (node)

    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <object>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_object (self, node):
    uid = int (self.__get_prop (node, 'uid') or '-1')
    name = self.__get_prop (node, 'name')
    value = self.mediator.call ('object.new')
    value.uid = uid

    node = node.children

    while node:
      if node.type == 'element' and node.name == 'attribute':
        item_name, item_value, datatype = self.load_attribute (node)
        value.set_attribute (item_name, item_value)

      elif node.type == 'element' and node.name == 'capability':
        value.add_capability (self.__get_prop (node, 'id'))

      elif node.type == 'element':
        item_name, item_value = self.load_item (node)
        setattr (value, item_name, item_value)

      node = node.next

    return name, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <pyobject>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_pyobject (self, node):
    name = self.__get_prop (node, 'name')
    value = PyObject ()
    node = node.children

    while node:
      item_name, item_value = self.load_item (node)
      if item_name:
        setattr (value, item_name, item_value)
      node = node.next

    return name, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load any item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_item (self, node):
    name, value = None, None

    if node.type == 'element' and node.name == 'attribute':
      name, value, datatype = self.load_attribute (node)

    elif node.type == 'element' and node.name == 'list':
      name, value = self.load_list (node)

    elif node.type == 'element' and node.name == 'set':
      name, value = self.load_set (node)

    elif node.type == 'element' and node.name == 'dict':
      name, value = self.load_dict (node)

    elif node.type == 'element' and node.name == 'object':
      name, value = self.load_object (node)

    elif node.type == 'element' and node.name == 'pyobject':
      name, value = self.load_pyobject (node)

    return name, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <attribute>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_attribute (self, node):
    name = self.__get_prop (node, 'name')
    value = self.__get_prop (node, 'value')
    datatype = self.__get_prop (node, 'datatype')

    if datatype == 'int':
      value = int (value)

    elif datatype == 'bool':
      value = (value == "true")

    elif datatype == 'float':
      value = float (value)

    elif datatype == 'NoneType':
      value = None

    return name, value, datatype

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <list>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_list (self, node):
    name = self.__get_prop (node, 'name')
    value = []

    node = node.children

    while node:
      if node.type == 'element':
        item_name, item_value = self.load_item (node)
        value.append (item_value)
      node = node.next

    return name, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <set>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_set (self, node):
    name = self.__get_prop (node, 'name')
    value = set ()

    node = node.children

    while node:
      if node.type == 'element':
        item_name, item_value = self.load_item (node)
        value.add (item_value)
      node = node.next

    return name, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <dict>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_dict (self, node):
    name = self.__get_prop (node, 'name')
    value = {}

    node = node.children

    while node:
      if node.type == 'element':
        item_name, item_value = self.load_item (node)
        value[item_name] = item_value
      node = node.next

    return name, value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save XML file
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save (self, path, value):
    doc = libxml2.newDoc ('1.0')
    node = self.save_item (value)
    doc.addChild (node)
    doc.saveFormatFileEnc (path, XML_ENCODING, 1)
    doc.freeDoc ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save any item
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_item (self, value):
    datatype = type (value).__name__
 
    if datatype == 'int':
      node = self.save_attribute (value, datatype)

    elif datatype == 'bool':
      node = self.save_attribute (value, datatype)

    elif datatype == 'float':
      node = self.save_attribute (value, datatype)

    elif datatype == 'str':
      node = self.save_attribute (value, datatype)

    elif datatype == 'unicode':
      node = self.save_attribute (value, datatype)

    elif datatype == 'NoneType':
      node = self.save_attribute (value, datatype)

    elif datatype == 'list':
      node = self.save_list (value)

    elif datatype == 'set':
      node = self.save_set (value)

    elif datatype == 'dict':
      node = self.save_dict (value)

    elif datatype == 'Object':
      node = self.save_object (value)

    else:
      node = self.save_pyobject (value)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <attribute>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_attribute (self, value, datatype):
    node = libxml2.newNode ('attribute')

    if datatype == 'bool':
      value = 'true' if value else 'false'

    if datatype != 'NoneType':
      self.__set_prop (node, 'value', str (value))

    if datatype != 'string' and datatype != 'unicode':
      self.__set_prop (node, 'datatype', datatype)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <list>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_list (self, value):
    node = libxml2.newNode ('list')

    for item in value:
      child = self.save_item (item)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <set>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_set (self, value):
    node = libxml2.newNode ('set')

    for item in value:
      child = self.save_item (item)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <dict>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_dict (self, value):
    node = libxml2.newNode ('dict')

    for name, value in value.iteritems ():
      child = self.save_item (value)
      self.__set_prop (child, 'name', name)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <pyobject>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_pyobject (self, value):
    node = libxml2.newNode ('pyobject')

    for name, value in vars (value).iteritems ():
      child = self.save_item (value)
      self.__set_prop (child, 'name', name)
      node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <object>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_object (self, value):
    node = libxml2.newNode ('object')
    if value.uid != -1:
      self.__set_prop (node, 'uid', str (value.uid))

    for item_name, item_value in value.iter_attributes ():
      child = self.save_item (item_value)
      self.__set_prop (child, 'name', item_name)
      node.addChild (child)

    for capability in value.iter_capabilities ():
      child = self.save_capability (capability)
      node.addChild (child)

    for item_name, item_value in vars (value).iteritems ():
      if item_name != 'uid' and not item_name.startswith ('_Object__'):
        child = self.save_item (item_value)
        self.__set_prop (child, 'name', item_name)
        node.addChild (child)

    return node

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save <capability>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def save_capability (self, value):
    node = libxml2.newNode ('capability')
    self.__set_prop (node, 'id', value)

    return node

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief SVG icon
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiIHN0YW5kYWxvbmU9Im5vIj8+Cjwh
LS0gQ3JlYXRlZCB3aXRoIElua3NjYXBlIChodHRwOi8vd3d3Lmlua3NjYXBlLm9yZy8pIC0tPgo8
c3ZnCiAgIHhtbG5zOmRjPSJodHRwOi8vcHVybC5vcmcvZGMvZWxlbWVudHMvMS4xLyIKICAgeG1s
bnM6Y2M9Imh0dHA6Ly9jcmVhdGl2ZWNvbW1vbnMub3JnL25zIyIKICAgeG1sbnM6cmRmPSJodHRw
Oi8vd3d3LnczLm9yZy8xOTk5LzAyLzIyLXJkZi1zeW50YXgtbnMjIgogICB4bWxuczpzdmc9Imh0
dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIgogICB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAw
MC9zdmciCiAgIHhtbG5zOnhsaW5rPSJodHRwOi8vd3d3LnczLm9yZy8xOTk5L3hsaW5rIgogICB4
bWxuczpzb2RpcG9kaT0iaHR0cDovL3NvZGlwb2RpLnNvdXJjZWZvcmdlLm5ldC9EVEQvc29kaXBv
ZGktMC5kdGQiCiAgIHhtbG5zOmlua3NjYXBlPSJodHRwOi8vd3d3Lmlua3NjYXBlLm9yZy9uYW1l
c3BhY2VzL2lua3NjYXBlIgogICBoZWlnaHQ9IjEyOC4wMDAwMHB0IgogICBpZD0ic3ZnMSIKICAg
aW5rc2NhcGU6dmVyc2lvbj0iMC40NiIKICAgc29kaXBvZGk6ZG9jYmFzZT0iL2hvbWUvYW5keS9E
ZXNrdG9wL2V0aXF1ZXR0ZS1pY29ucy0wLjQiCiAgIHNvZGlwb2RpOmRvY25hbWU9ImV4dGVuc2lv
bi14bWwtcGlja2xlLnN2ZyIKICAgc29kaXBvZGk6dmVyc2lvbj0iMC4zMiIKICAgd2lkdGg9IjEy
OC4wMDAwMHB0IgogICBpbmtzY2FwZTpvdXRwdXRfZXh0ZW5zaW9uPSJvcmcuaW5rc2NhcGUub3V0
cHV0LnN2Zy5pbmtzY2FwZSI+CiAgPG1ldGFkYXRhCiAgICAgaWQ9Im1ldGFkYXRhMyI+CiAgICA8
cmRmOlJERj4KICAgICAgPGNjOldvcmsKICAgICAgICAgcmRmOmFib3V0PSIiPgogICAgICAgIDxk
Yzp0aXRsZT5FdGlxdWV0dGUgSWNvbnM8L2RjOnRpdGxlPgogICAgICAgIDxkYzpkZXNjcmlwdGlv
biAvPgogICAgICAgIDxkYzpzdWJqZWN0PgogICAgICAgICAgPHJkZjpCYWc+CiAgICAgICAgICAg
IDxyZGY6bGk+aGFzaDwvcmRmOmxpPgogICAgICAgICAgICA8cmRmOmxpIC8+CiAgICAgICAgICAg
IDxyZGY6bGk+Y29tcHV0ZXI8L3JkZjpsaT4KICAgICAgICAgICAgPHJkZjpsaT5pY29uczwvcmRm
OmxpPgogICAgICAgICAgICA8cmRmOmxpPnRoZW1lPC9yZGY6bGk+CiAgICAgICAgICA8L3JkZjpC
YWc+CiAgICAgICAgPC9kYzpzdWJqZWN0PgogICAgICAgIDxkYzpwdWJsaXNoZXI+CiAgICAgICAg
ICA8Y2M6QWdlbnQKICAgICAgICAgICAgIHJkZjphYm91dD0iaHR0cDovL3d3dy5vcGVuY2xpcGFy
dC5vcmciPgogICAgICAgICAgICA8ZGM6dGl0bGU+QW5keSBGaXR6c2ltb248L2RjOnRpdGxlPgog
ICAgICAgICAgPC9jYzpBZ2VudD4KICAgICAgICA8L2RjOnB1Ymxpc2hlcj4KICAgICAgICA8ZGM6
Y3JlYXRvcj4KICAgICAgICAgIDxjYzpBZ2VudD4KICAgICAgICAgICAgPGRjOnRpdGxlPkFuZHkg
Rml0enNpbW9uPC9kYzp0aXRsZT4KICAgICAgICAgIDwvY2M6QWdlbnQ+CiAgICAgICAgPC9kYzpj
cmVhdG9yPgogICAgICAgIDxkYzpyaWdodHM+CiAgICAgICAgICA8Y2M6QWdlbnQ+CiAgICAgICAg
ICAgIDxkYzp0aXRsZT5BbmR5IEZpdHpzaW1vbjwvZGM6dGl0bGU+CiAgICAgICAgICA8L2NjOkFn
ZW50PgogICAgICAgIDwvZGM6cmlnaHRzPgogICAgICAgIDxkYzpkYXRlIC8+CiAgICAgICAgPGRj
OmZvcm1hdD5pbWFnZS9zdmcreG1sPC9kYzpmb3JtYXQ+CiAgICAgICAgPGRjOnR5cGUKICAgICAg
ICAgICByZGY6cmVzb3VyY2U9Imh0dHA6Ly9wdXJsLm9yZy9kYy9kY21pdHlwZS9TdGlsbEltYWdl
IiAvPgogICAgICAgIDxjYzpsaWNlbnNlCiAgICAgICAgICAgcmRmOnJlc291cmNlPSJodHRwOi8v
d2ViLnJlc291cmNlLm9yZy9jYy9QdWJsaWNEb21haW4iIC8+CiAgICAgICAgPGRjOmxhbmd1YWdl
PmVuPC9kYzpsYW5ndWFnZT4KICAgICAgPC9jYzpXb3JrPgogICAgICA8Y2M6TGljZW5zZQogICAg
ICAgICByZGY6YWJvdXQ9Imh0dHA6Ly93ZWIucmVzb3VyY2Uub3JnL2NjL1B1YmxpY0RvbWFpbiI+
CiAgICAgICAgPGNjOnBlcm1pdHMKICAgICAgICAgICByZGY6cmVzb3VyY2U9Imh0dHA6Ly93ZWIu
cmVzb3VyY2Uub3JnL2NjL1JlcHJvZHVjdGlvbiIgLz4KICAgICAgICA8Y2M6cGVybWl0cwogICAg
ICAgICAgIHJkZjpyZXNvdXJjZT0iaHR0cDovL3dlYi5yZXNvdXJjZS5vcmcvY2MvRGlzdHJpYnV0
aW9uIiAvPgogICAgICAgIDxjYzpwZXJtaXRzCiAgICAgICAgICAgcmRmOnJlc291cmNlPSJodHRw
Oi8vd2ViLnJlc291cmNlLm9yZy9jYy9EZXJpdmF0aXZlV29ya3MiIC8+CiAgICAgIDwvY2M6TGlj
ZW5zZT4KICAgIDwvcmRmOlJERj4KICA8L21ldGFkYXRhPgogIDxkZWZzCiAgICAgaWQ9ImRlZnMz
Ij4KICAgIDxsaW5lYXJHcmFkaWVudAogICAgICAgaW5rc2NhcGU6Y29sbGVjdD0iYWx3YXlzIgog
ICAgICAgaWQ9ImxpbmVhckdyYWRpZW50MzIyNyI+CiAgICAgIDxzdG9wCiAgICAgICAgIHN0eWxl
PSJzdG9wLWNvbG9yOiMwMDAwMDA7c3RvcC1vcGFjaXR5OjE7IgogICAgICAgICBvZmZzZXQ9IjAi
CiAgICAgICAgIGlkPSJzdG9wMzIyOSIgLz4KICAgICAgPHN0b3AKICAgICAgICAgc3R5bGU9InN0
b3AtY29sb3I6IzAwMDAwMDtzdG9wLW9wYWNpdHk6MDsiCiAgICAgICAgIG9mZnNldD0iMSIKICAg
ICAgICAgaWQ9InN0b3AzMjMxIiAvPgogICAgPC9saW5lYXJHcmFkaWVudD4KICAgIDxsaW5lYXJH
cmFkaWVudAogICAgICAgaW5rc2NhcGU6Y29sbGVjdD0iYWx3YXlzIgogICAgICAgaWQ9ImxpbmVh
ckdyYWRpZW50MzIxOSI+CiAgICAgIDxzdG9wCiAgICAgICAgIHN0eWxlPSJzdG9wLWNvbG9yOiMw
MDAwMDA7c3RvcC1vcGFjaXR5OjE7IgogICAgICAgICBvZmZzZXQ9IjAiCiAgICAgICAgIGlkPSJz
dG9wMzIyMSIgLz4KICAgICAgPHN0b3AKICAgICAgICAgc3R5bGU9InN0b3AtY29sb3I6IzAwMDAw
MDtzdG9wLW9wYWNpdHk6MDsiCiAgICAgICAgIG9mZnNldD0iMSIKICAgICAgICAgaWQ9InN0b3Az
MjIzIiAvPgogICAgPC9saW5lYXJHcmFkaWVudD4KICAgIDxpbmtzY2FwZTpwZXJzcGVjdGl2ZQog
ICAgICAgc29kaXBvZGk6dHlwZT0iaW5rc2NhcGU6cGVyc3AzZCIKICAgICAgIGlua3NjYXBlOnZw
X3g9IjAgOiA4MCA6IDEiCiAgICAgICBpbmtzY2FwZTp2cF95PSIwIDogMTAwMCA6IDAiCiAgICAg
ICBpbmtzY2FwZTp2cF96PSIxNjAgOiA4MCA6IDEiCiAgICAgICBpbmtzY2FwZTpwZXJzcDNkLW9y
aWdpbj0iODAgOiA1My4zMzMzMzMgOiAxIgogICAgICAgaWQ9InBlcnNwZWN0aXZlNTYiIC8+CiAg
ICA8bGluZWFyR3JhZGllbnQKICAgICAgIGlkPSJsaW5lYXJHcmFkaWVudDE1MDciPgogICAgICA8
c3RvcAogICAgICAgICBpZD0ic3RvcDE1MDgiCiAgICAgICAgIG9mZnNldD0iMC4wMDAwMDAwIgog
ICAgICAgICBzdHlsZT0ic3RvcC1jb2xvcjojMDAwMDAwO3N0b3Atb3BhY2l0eTowLjA5NTUwNTYx
ODsiIC8+CiAgICAgIDxzdG9wCiAgICAgICAgIGlkPSJzdG9wMTUxMCIKICAgICAgICAgb2Zmc2V0
PSIxLjAwMDAwMDAiCiAgICAgICAgIHN0eWxlPSJzdG9wLWNvbG9yOiMwMDAwMDA7c3RvcC1vcGFj
aXR5OjAuMDAwMDAwMDsiIC8+CiAgICA8L2xpbmVhckdyYWRpZW50PgogICAgPGxpbmVhckdyYWRp
ZW50CiAgICAgICBpZD0ibGluZWFyR3JhZGllbnQ4NTIiCiAgICAgICB4bGluazpocmVmPSIjbGlu
ZWFyR3JhZGllbnQxNTA3IiAvPgogICAgPGxpbmVhckdyYWRpZW50CiAgICAgICBpZD0ibGluZWFy
R3JhZGllbnQ4NDQiCiAgICAgICB4bGluazpocmVmPSIjbGluZWFyR3JhZGllbnQxNTA3IiAvPgog
ICAgPGxpbmVhckdyYWRpZW50CiAgICAgICBpZD0ibGluZWFyR3JhZGllbnQ4NDUiCiAgICAgICB4
MT0iMS4zODMzMzMzIgogICAgICAgeDI9IjAuMTA4MzMzMzMiCiAgICAgICB4bGluazpocmVmPSIj
bGluZWFyR3JhZGllbnQ4NDEiCiAgICAgICB5MT0iMC40OTAxOTYwOCIKICAgICAgIHkyPSIwLjUw
NDkwMTk1IiAvPgogICAgPGxpbmVhckdyYWRpZW50CiAgICAgICBpZD0ibGluZWFyR3JhZGllbnQ4
NDAiCiAgICAgICB4MT0iLTAuMjIzNDg0ODQiCiAgICAgICB4Mj0iMC41OTQ2OTcwMCIKICAgICAg
IHhsaW5rOmhyZWY9IiNsaW5lYXJHcmFkaWVudDg1MyIKICAgICAgIHkxPSIwLjM4MjM1Mjk1Igog
ICAgICAgeTI9IjAuNDY1Njg2MjYiIC8+CiAgICA8bGluZWFyR3JhZGllbnQKICAgICAgIGlkPSJs
aW5lYXJHcmFkaWVudDg0OSIKICAgICAgIHgxPSIwLjAxMDU2MzM4MCIKICAgICAgIHgyPSIxLjIy
ODg3MzMiCiAgICAgICB4bGluazpocmVmPSIjbGluZWFyR3JhZGllbnQ4NDYiCiAgICAgICB5MT0i
MC40MzIyOTE2NiIKICAgICAgIHkyPSIwLjQ2MzU0MTY2IiAvPgogICAgPHJhZGlhbEdyYWRpZW50
CiAgICAgICBjeD0iMC41MDAwMDAwMCIKICAgICAgIGN5PSIwLjUwMDAwMDAwIgogICAgICAgZng9
IjAuNTAwMDAwMDAiCiAgICAgICBmeT0iMC4xNDk0MjUyOCIKICAgICAgIGlkPSJyYWRpYWxHcmFk
aWVudDg2NCIKICAgICAgIHI9IjAuNTAwMDAwMDAiCiAgICAgICB4bGluazpocmVmPSIjbGluZWFy
R3JhZGllbnQ4NTMiIC8+CiAgICA8bGluZWFyR3JhZGllbnQKICAgICAgIGlkPSJsaW5lYXJHcmFk
aWVudDg0MSI+CiAgICAgIDxzdG9wCiAgICAgICAgIGlkPSJzdG9wODQyIgogICAgICAgICBvZmZz
ZXQ9IjAuMDAwMDAwMDAiCiAgICAgICAgIHN0eWxlPSJzdG9wLWNvbG9yOiNmZmZmZmY7c3RvcC1v
cGFjaXR5OjEuMDAwMDAwMDsiIC8+CiAgICAgIDxzdG9wCiAgICAgICAgIGlkPSJzdG9wODQzIgog
ICAgICAgICBvZmZzZXQ9IjEuMDAwMDAwMCIKICAgICAgICAgc3R5bGU9InN0b3AtY29sb3I6I2Zm
ZmZmZjtzdG9wLW9wYWNpdHk6MC4wMDAwMDAwMDsiIC8+CiAgICA8L2xpbmVhckdyYWRpZW50Pgog
ICAgPGxpbmVhckdyYWRpZW50CiAgICAgICBpZD0ibGluZWFyR3JhZGllbnQxMjkwIj4KICAgICAg
PHN0b3AKICAgICAgICAgaWQ9InN0b3AxMjkxIgogICAgICAgICBvZmZzZXQ9IjAuMDAwMDAwMCIK
ICAgICAgICAgc3R5bGU9InN0b3AtY29sb3I6I2IyYTI2OTtzdG9wLW9wYWNpdHk6MS4wMDAwMDAw
OyIgLz4KICAgICAgPHN0b3AKICAgICAgICAgaWQ9InN0b3AxMjkyIgogICAgICAgICBvZmZzZXQ9
IjEuMDAwMDAwMCIKICAgICAgICAgc3R5bGU9InN0b3AtY29sb3I6IzZkNWIxODtzdG9wLW9wYWNp
dHk6MS4wMDAwMDAwOyIgLz4KICAgIDwvbGluZWFyR3JhZGllbnQ+CiAgICA8bGluZWFyR3JhZGll
bnQKICAgICAgIGlkPSJsaW5lYXJHcmFkaWVudDg2MCIKICAgICAgIHgxPSIwLjQ3NTM1MjExIgog
ICAgICAgeDI9IjAuNTAwMDAwMDAiCiAgICAgICB4bGluazpocmVmPSIjbGluZWFyR3JhZGllbnQx
MjkwIgogICAgICAgeTE9IjAuODEwODEwODAiCiAgICAgICB5Mj0iLTAuNzQzMjQzMjIiIC8+CiAg
ICA8bGluZWFyR3JhZGllbnQKICAgICAgIGlkPSJsaW5lYXJHcmFkaWVudDg1MSIKICAgICAgIHgx
PSIwLjM5Nzg4NzMyIgogICAgICAgeDI9IjAuODA5ODU5MTYiCiAgICAgICB4bGluazpocmVmPSIj
bGluZWFyR3JhZGllbnQ4NDYiCiAgICAgICB5MT0iMC4zMjIyMjIyMyIKICAgICAgIHkyPSIwLjM1
NTU1NTU2IiAvPgogICAgPGxpbmVhckdyYWRpZW50CiAgICAgICBpZD0ibGluZWFyR3JhZGllbnQ4
NTgiCiAgICAgICB4MT0iMC42NDI4NTcxMyIKICAgICAgIHgyPSIwLjU3MTQyODYwIgogICAgICAg
eGxpbms6aHJlZj0iI2xpbmVhckdyYWRpZW50ODQ2IgogICAgICAgeTE9IjEuMjY0NzA1OSIKICAg
ICAgIHkyPSIwLjA0OTAxOTYwOSIgLz4KICAgIDxsaW5lYXJHcmFkaWVudAogICAgICAgaWQ9Imxp
bmVhckdyYWRpZW50ODQ2Ij4KICAgICAgPHN0b3AKICAgICAgICAgaWQ9InN0b3A4NDciCiAgICAg
ICAgIG9mZnNldD0iMC4wMDAwMDAwMCIKICAgICAgICAgc3R5bGU9InN0b3AtY29sb3I6I2U3ZTdl
NztzdG9wLW9wYWNpdHk6MS4wMDAwMDAwOyIgLz4KICAgICAgPHN0b3AKICAgICAgICAgaWQ9InN0
b3A4NDgiCiAgICAgICAgIG9mZnNldD0iMS4wMDAwMDAwIgogICAgICAgICBzdHlsZT0ic3RvcC1j
b2xvcjojYTVhNWE1O3N0b3Atb3BhY2l0eToxLjAwMDAwMDA7IiAvPgogICAgPC9saW5lYXJHcmFk
aWVudD4KICAgIDxsaW5lYXJHcmFkaWVudAogICAgICAgaWQ9ImxpbmVhckdyYWRpZW50ODUwIgog
ICAgICAgeGxpbms6aHJlZj0iI2xpbmVhckdyYWRpZW50ODQ2IiAvPgogICAgPGxpbmVhckdyYWRp
ZW50CiAgICAgICBpZD0ibGluZWFyR3JhZGllbnQ4NTkiCiAgICAgICB4MT0iMS40NjQ3ODg3Igog
ICAgICAgeDI9IjAuMjY0MDg0NTIiCiAgICAgICB4bGluazpocmVmPSIjbGluZWFyR3JhZGllbnQ4
NTMiCiAgICAgICB5MT0iLTEuMTQ4NjQ4NiIKICAgICAgIHkyPSIxLjI5MDU0MDYiIC8+CiAgICA8
bGluZWFyR3JhZGllbnQKICAgICAgIGlkPSJsaW5lYXJHcmFkaWVudDg1MyI+CiAgICAgIDxzdG9w
CiAgICAgICAgIGlkPSJzdG9wODU0IgogICAgICAgICBvZmZzZXQ9IjAuMDAwMDAwMDAiCiAgICAg
ICAgIHN0eWxlPSJzdG9wLWNvbG9yOiMwMDAwMDA7c3RvcC1vcGFjaXR5OjAuMjk3NTIwNjc7IiAv
PgogICAgICA8c3RvcAogICAgICAgICBpZD0ic3RvcDg1NSIKICAgICAgICAgb2Zmc2V0PSIxLjAw
MDAwMDAiCiAgICAgICAgIHN0eWxlPSJzdG9wLWNvbG9yOiMwMDAwMDA7c3RvcC1vcGFjaXR5OjAu
MDAwMDAwMDA7IiAvPgogICAgPC9saW5lYXJHcmFkaWVudD4KICAgIDxyYWRpYWxHcmFkaWVudAog
ICAgICAgY3g9IjAuNTAwMDAwMDAiCiAgICAgICBjeT0iMC41MDAwMDAwMCIKICAgICAgIGZ4PSIw
LjUwNzA0MjIzIgogICAgICAgZnk9IjAuMjk4ODUwNTciCiAgICAgICBpZD0icmFkaWFsR3JhZGll
bnQ4NjEiCiAgICAgICByPSIwLjUwMDAwMDAwIgogICAgICAgeGxpbms6aHJlZj0iI2xpbmVhckdy
YWRpZW50ODUzIiAvPgogICAgPGxpbmVhckdyYWRpZW50CiAgICAgICBpZD0ibGluZWFyR3JhZGll
bnQxNTAxIj4KICAgICAgPHN0b3AKICAgICAgICAgaWQ9InN0b3AxNTAyIgogICAgICAgICBvZmZz
ZXQ9IjAuMDAwMDAwMCIKICAgICAgICAgc3R5bGU9InN0b3AtY29sb3I6I2ZmZmZmZjtzdG9wLW9w
YWNpdHk6MS4wMDAwMDAwOyIgLz4KICAgICAgPHN0b3AKICAgICAgICAgaWQ9InN0b3AxNTA0Igog
ICAgICAgICBvZmZzZXQ9IjEuMDAwMDAwMCIKICAgICAgICAgc3R5bGU9InN0b3AtY29sb3I6I2Zm
ZmZmZjtzdG9wLW9wYWNpdHk6MC4wMDAwMDAwOyIgLz4KICAgIDwvbGluZWFyR3JhZGllbnQ+CiAg
ICA8bGluZWFyR3JhZGllbnQKICAgICAgIGlkPSJsaW5lYXJHcmFkaWVudDE0OTIiPgogICAgICA8
c3RvcAogICAgICAgICBpZD0ic3RvcDE0OTMiCiAgICAgICAgIG9mZnNldD0iMC4wMDAwMDAwIgog
ICAgICAgICBzdHlsZT0ic3RvcC1jb2xvcjojY2JjYmNiO3N0b3Atb3BhY2l0eToxLjAwMDAwMDA7
IiAvPgogICAgICA8c3RvcAogICAgICAgICBpZD0ic3RvcDE0OTYiCiAgICAgICAgIG9mZnNldD0i
MC4zNDkyMzA3NyIKICAgICAgICAgc3R5bGU9InN0b3AtY29sb3I6I2YwZjBmMDtzdG9wLW9wYWNp
dHk6MS4wMDAwMDAwOyIgLz4KICAgICAgPHN0b3AKICAgICAgICAgaWQ9InN0b3AxNDk0IgogICAg
ICAgICBvZmZzZXQ9IjEuMDAwMDAwMCIKICAgICAgICAgc3R5bGU9InN0b3AtY29sb3I6I2UyZTJl
MjtzdG9wLW9wYWNpdHk6MS4wMDAwMDAwOyIgLz4KICAgIDwvbGluZWFyR3JhZGllbnQ+CiAgICA8
bGluZWFyR3JhZGllbnQKICAgICAgIGlkPSJsaW5lYXJHcmFkaWVudDE0OTUiCiAgICAgICB4MT0i
MC45MjMwNzY5MyIKICAgICAgIHgyPSIwLjE0NTI5OTE1IgogICAgICAgeGxpbms6aHJlZj0iI2xp
bmVhckdyYWRpZW50MTQ5MiIKICAgICAgIHkxPSIwLjE2NDA2MjUwIgogICAgICAgeTI9IjEuMTcx
ODc1MCIgLz4KICAgIDxsaW5lYXJHcmFkaWVudAogICAgICAgaWQ9ImxpbmVhckdyYWRpZW50MTQ5
NyIKICAgICAgIHgxPSIwLjYzMjQ3ODY1IgogICAgICAgeDI9Ii0wLjM3NjA2ODM4IgogICAgICAg
eGxpbms6aHJlZj0iI2xpbmVhckdyYWRpZW50MTQ5MiIKICAgICAgIHkxPSIwLjMyODEyNTAwIgog
ICAgICAgeTI9IjEuMzI4MTI1MCIgLz4KICAgIDxsaW5lYXJHcmFkaWVudAogICAgICAgaWQ9Imxp
bmVhckdyYWRpZW50MTQ5OSIKICAgICAgIHgxPSIwLjg1ODI2NzcyIgogICAgICAgeDI9IjAuMDYy
OTkyMTI2IgogICAgICAgeGxpbms6aHJlZj0iI2xpbmVhckdyYWRpZW50MTUwMSIKICAgICAgIHkx
PSIwLjE0MDYyNTAwIgogICAgICAgeTI9IjAuNTQ2ODc1MDAiIC8+CiAgICA8bGluZWFyR3JhZGll
bnQKICAgICAgIGlkPSJsaW5lYXJHcmFkaWVudDE1MDYiCiAgICAgICB4MT0iMC4wNTIxNzM5MTMi
CiAgICAgICB4Mj0iMC43ODI2MDg2OSIKICAgICAgIHhsaW5rOmhyZWY9IiNsaW5lYXJHcmFkaWVu
dDE1MDciCiAgICAgICB5MT0iMC45NzY1NjI1MCIKICAgICAgIHkyPSIwLjAwNzgxMjUwMDAiIC8+
CiAgICA8bGluZWFyR3JhZGllbnQKICAgICAgIGlkPSJsaW5lYXJHcmFkaWVudDE1NTYiCiAgICAg
ICB4MT0iMC4zMTExMTExMiIKICAgICAgIHgyPSIwLjYyMjIyMjI0IgogICAgICAgeGxpbms6aHJl
Zj0iI2xpbmVhckdyYWRpZW50MTUwNyIKICAgICAgIHkxPSItMC41NjI1MDAwMCIKICAgICAgIHky
PSIwLjc5Njg3NTAwIiAvPgogICAgPHJhZGlhbEdyYWRpZW50CiAgICAgICBjeD0iMC41MDAwMDAw
MCIKICAgICAgIGN5PSIwLjg5Mjg1NzEzIgogICAgICAgZng9IjAuNTQxMTc2NTAiCiAgICAgICBm
eT0iMy41MjAwMDAwIgogICAgICAgaWQ9InJhZGlhbEdyYWRpZW50ODU2IgogICAgICAgcj0iMC41
NDYwNjY3MCIKICAgICAgIHhsaW5rOmhyZWY9IiNsaW5lYXJHcmFkaWVudDg0MSIgLz4KICAgIDxs
aW5lYXJHcmFkaWVudAogICAgICAgaWQ9ImxpbmVhckdyYWRpZW50MTk0NCIKICAgICAgIHhsaW5r
OmhyZWY9IiNsaW5lYXJHcmFkaWVudDg0MSIgLz4KICAgIDxsaW5lYXJHcmFkaWVudAogICAgICAg
aW5rc2NhcGU6Y29sbGVjdD0iYWx3YXlzIgogICAgICAgeGxpbms6aHJlZj0iI2xpbmVhckdyYWRp
ZW50MzIxOSIKICAgICAgIGlkPSJsaW5lYXJHcmFkaWVudDMyMjUiCiAgICAgICB4MT0iMTE2LjEx
MDA5IgogICAgICAgeTE9IjEyMi40NTM3MiIKICAgICAgIHgyPSIxMzguMTA2MDkiCiAgICAgICB5
Mj0iMTIyLjQ1MzcyIgogICAgICAgZ3JhZGllbnRVbml0cz0idXNlclNwYWNlT25Vc2UiIC8+CiAg
ICA8bGluZWFyR3JhZGllbnQKICAgICAgIGlua3NjYXBlOmNvbGxlY3Q9ImFsd2F5cyIKICAgICAg
IHhsaW5rOmhyZWY9IiNsaW5lYXJHcmFkaWVudDMyMjciCiAgICAgICBpZD0ibGluZWFyR3JhZGll
bnQzMjMzIgogICAgICAgeDE9IjExNS42MTAwOSIKICAgICAgIHkxPSIxMjIuNDUzNzIiCiAgICAg
ICB4Mj0iMTM4LjYwNjA5IgogICAgICAgeTI9IjEyMi40NTM3MiIKICAgICAgIGdyYWRpZW50VW5p
dHM9InVzZXJTcGFjZU9uVXNlIiAvPgogIDwvZGVmcz4KICA8c29kaXBvZGk6bmFtZWR2aWV3CiAg
ICAgYm9yZGVyY29sb3I9IiM2NjY2NjYiCiAgICAgYm9yZGVyb3BhY2l0eT0iMS4wIgogICAgIGlk
PSJiYXNlIgogICAgIGlua3NjYXBlOmN4PSIxMzAuMjI5MjgiCiAgICAgaW5rc2NhcGU6Y3k9Ijgw
IgogICAgIGlua3NjYXBlOnBhZ2VvcGFjaXR5PSIwLjAiCiAgICAgaW5rc2NhcGU6cGFnZXNoYWRv
dz0iMiIKICAgICBpbmtzY2FwZTp3aW5kb3ctaGVpZ2h0PSI3OTIiCiAgICAgaW5rc2NhcGU6d2lu
ZG93LXdpZHRoPSIxMDIxIgogICAgIGlua3NjYXBlOndpbmRvdy14PSIxMjQiCiAgICAgaW5rc2Nh
cGU6d2luZG93LXk9IjIwIgogICAgIGlua3NjYXBlOnpvb209IjMuNTQzNzUiCiAgICAgcGFnZWNv
bG9yPSIjZmZmZmZmIgogICAgIHNob3dncmlkPSJmYWxzZSIKICAgICBpbmtzY2FwZTpjdXJyZW50
LWxheWVyPSJzdmcxIiAvPgogIDxwYXRoCiAgICAgZD0iTSAxNy4xNTkzODQsNi4zMjkyODk0IEwg
MTcuMTU5Mzg0LDQzLjA2ODUyNiBMIDE3LjE1OTM4NCw3OS44MDc3NjggTCAxNy4xNTkzODQsMTE2
LjU0NzAwIEwgMTcuMTU5Mzg0LDE1My4yODYyNCBMIDUwLjY4NzAyNCwxNTMuMjg2MjQgTCA4NC4y
MTQ2NjQsMTUzLjI4NjI0IEwgMTE3Ljc0MjMwLDE1My4yODYyNCBMIDE1MS4yNjk5NCwxNTMuMjg2
MjQgTCAxNTEuMjY5OTQsMTE2LjU0NzAwIEwgMTUxLjI2OTk0LDc5LjgwNzc2OCBMIDE1MS4yNjk5
NCw0My4wNjg1MjYgTCAxMTcuNzQyMzAsNi4zMjkyODk0IEwgODQuMjE0NjY0LDYuMzI5Mjg5NCBM
IDUwLjY4NzAyNCw2LjMyOTI4OTQgTCAxNy4xNTkzODQsNi4zMjkyODk0IHogIgogICAgIGlkPSJw
YXRoOTMwIgogICAgIHNvZGlwb2RpOm5vZGV0eXBlcz0iY2NjY2NjY2NjY2NjY2NjYyIKICAgICBz
dHlsZT0iZmlsbDojZmZmZmZmO2ZpbGwtcnVsZTpldmVub2RkO3N0cm9rZS13aWR0aDowLjQyNjQ5
MDQzO3N0cm9rZS1vcGFjaXR5OjAuMzY0Nzc5ODY7IiAvPgogIDxnCiAgICAgaWQ9ImcxNTUyIgog
ICAgIHN0eWxlPSJmaWxsOiMwMDAwMDA7ZmlsbC1vcGFjaXR5OjAuMDY5MTgyMzQ0OyIKICAgICB0
cmFuc2Zvcm09Im1hdHJpeCgwLjMwNDE3MSwwLjAwMDAwMCwwLjAwMDAwMCwwLjI5NzU3MiwtMzYu
NzAzOTksLTExMi40ODgwKSI+CiAgICA8cGF0aAogICAgICAgZD0iTSAxNzMuMzU5NTksNDA4Ljgx
MjYwIEwgMTczLjM1OTU5LDUzMS4wNzM2MCBMIDE3My4zNTk1OSw2NTMuMzM0NjAgTCAxNzMuMzU5
NTksNzc1LjU5NTYwIEwgMTczLjM1OTU5LDg5Ny44NTY2MCBMIDI4Mi4xMjE4Nyw4OTcuODU2NjAg
TCAzOTAuODg0MTcsODk3Ljg1NjYwIEwgNDk5LjY0NjQ2LDg5Ny44NTY2MCBMIDYwOC40MDg3NCw4
OTcuODU2NjAgTCA2MDguNDA4NzQsNzc1LjU5NTYwIEwgNjA4LjQwODc0LDY1My4zMzQ2MCBMIDYw
OC40MDg3NCw1MzEuMDczNjAgTCA0OTkuNjQ2NDYsNDA4LjgxMjYwIEwgMzkwLjg4NDE3LDQwOC44
MTI2MCBMIDI4Mi4xMjE4Nyw0MDguODEyNjAgTCAxNzMuMzU5NTksNDA4LjgxMjYwIHogIgogICAg
ICAgaWQ9InBhdGgxNTA1IgogICAgICAgc29kaXBvZGk6bm9kZXR5cGVzPSJjY2NjY2NjY2NjY2Nj
Y2NjIgogICAgICAgc3R5bGU9ImZpbGwtb3BhY2l0eTowLjA2OTE4MjM0NDtmaWxsLXJ1bGU6ZXZl
bm9kZDtzdHJva2Utd2lkdGg6MC45NTQwNzQ5NXB0O2ZpbGw6IzAwMDAwMDsiIC8+CiAgICA8cGF0
aAogICAgICAgZD0iTSAxNzAuMTI1MDAsNDA3LjE4NzUwIEMgMTcwLjEyNTAwLDU3MS44MjI5MiAx
NzAuMTI1MDAsNzM2LjQ1ODMzIDE3MC4xMjUwMCw5MDEuMDkzNzUgQyAzMTcuMzAyMDgsOTAxLjA5
Mzc1IDQ2NC40NzkxNyw5MDEuMDkzNzUgNjExLjY1NjI1LDkwMS4wOTM3NSBDIDYxMS42NTYyNSw3
NzcuMzMzMzMgNjExLjY1NjI1LDY1My41NzI5MiA2MTEuNjU2MjUsNTI5LjgxMjUwIEMgNTc0Ljgw
OTE4LDQ4OC4zODk3OCA1MzcuOTc2MDQsNDQ2Ljk1MzM1IDUwMS4wOTM3NSw0MDUuNTYyNTAgQyAz
OTAuNzcwODMsNDA1LjU2MjUwIDI4MC40NDc5Miw0MDUuNTYyNTAgMTcwLjEyNTAwLDQwNS41NjI1
MCBDIDE3MC4xMjUwMCw0MDUuODk1ODMgMTcwLjEyNTAwLDQwNi45NzkxNyAxNzAuMTI1MDAsNDA3
LjE4NzUwIHogIgogICAgICAgaWQ9InBhdGgxNTM4IgogICAgICAgc3R5bGU9ImZpbGwtb3BhY2l0
eTowLjA2OTE4MjM0NDtmaWxsLXJ1bGU6ZXZlbm9kZDtzdHJva2Utd2lkdGg6MC45NTQwNzQ5NXB0
O2ZpbGw6IzAwMDAwMDsiIC8+CiAgICA8cGF0aAogICAgICAgZD0iTSAxNjYuODc1MDAsNDAzLjkz
NzUwIEMgMTY2Ljg3NTAwLDU3MC43Mzk1OCAxNjYuODc1MDAsNzM3LjU0MTY3IDE2Ni44NzUwMCw5
MDQuMzQzNzUgQyAzMTYuMjE4NzUsOTA0LjM0Mzc1IDQ2NS41NjI1MCw5MDQuMzQzNzUgNjE0Ljkw
NjI1LDkwNC4zNDM3NSBDIDYxNC45MDYyNSw3NzkuMDgzMzMgNjE0LjkwNjI1LDY1My44MjI5MiA2
MTQuOTA2MjUsNTI4LjU2MjUwIEMgNTc3LjQ1NTE3LDQ4Ni40NzI5OCA1NDAuMDE3NTQsNDQ0LjM3
MDE3IDUwMi41MzEyNSw0MDIuMzEyNTAgQyAzOTAuNjQ1ODMsNDAyLjMxMjUwIDI3OC43NjA0Miw0
MDIuMzEyNTAgMTY2Ljg3NTAwLDQwMi4zMTI1MCBDIDE2Ni44NzUwMCw0MDIuNjQ1ODMgMTY2Ljg3
NTAwLDQwMy43MjkxNyAxNjYuODc1MDAsNDAzLjkzNzUwIHogIgogICAgICAgaWQ9InBhdGgxNTQz
IgogICAgICAgc3R5bGU9ImZpbGwtb3BhY2l0eTowLjA2OTE4MjM0NDtmaWxsLXJ1bGU6ZXZlbm9k
ZDtzdHJva2Utd2lkdGg6MC45NTQwNzQ5NXB0O2ZpbGw6IzAwMDAwMDsiIC8+CiAgPC9nPgogIDxw
YXRoCiAgICAgZD0iTSAyOTcuOTkwMzQsMTM2Ljc0MTU0IEwgMjk3Ljk5MDM0LDI2MC41NzIzNCBM
IDI5Ny45OTAzNCwzODQuNDAzMTUgTCAyOTcuOTkwMzQsNTA4LjIzMzk1IEwgMjk3Ljk5MDM0LDYz
Mi4wNjQ3NSBMIDQxMC45OTYzNCw2MzIuMDY0NzUgTCA1MjQuMDAyMzUsNjMyLjA2NDc1IEwgNjM3
LjAwODM1LDYzMi4wNjQ3NSBMIDc1MC4wMTQzNSw2MzIuMDY0NzUgTCA3NTAuMDE0MzUsNTA4LjIz
Mzk1IEwgNzUwLjAxNDM1LDM4NC40MDMxNSBMIDc1MC4wMTQzNSwyNjAuNTcyMzQgTCA2MzcuMDA4
MzUsMTM2Ljc0MTU0IEwgNTI0LjAwMjM1LDEzNi43NDE1NCBMIDQxMC45OTYzNCwxMzYuNzQxNTQg
TCAyOTcuOTkwMzQsMTM2Ljc0MTU0IHogIgogICAgIGlkPSJyZWN0OTAwIgogICAgIHNvZGlwb2Rp
Om5vZGV0eXBlcz0iY2NjY2NjY2NjY2NjY2NjYyIKICAgICBzdHlsZT0iZmlsbDp1cmwoI2xpbmVh
ckdyYWRpZW50MTQ5NSk7ZmlsbC1vcGFjaXR5OjE7ZmlsbC1ydWxlOmV2ZW5vZGQ7c3Ryb2tlOiMw
MDAwMDA7c3Ryb2tlLXdpZHRoOjEuNDM3NTAwMDtzdHJva2Utb3BhY2l0eTowLjM2NDc3OTg2OyIK
ICAgICB0cmFuc2Zvcm09Im1hdHJpeCgwLjI5NjY4OSwwLjAwMDAwMCwwLjAwMDAwMCwwLjI5NjY4
OSwtNzEuMTk2MDEsLTM0LjA1ODI1KSIgLz4KICA8cGF0aAogICAgIGQ9Ik0gNzUwLjAxNDM1LDI2
MC41NzIzNCBDIDcxNS45NDQ2MCwyNTAuNDg2MDIgNjcxLjc5Nzg4LDI1MS45MTYyNCA2MzguNDQ3
OTIsMjU3LjY2NTE2IEMgNjQ0LjIwNjE4LDIyMC43MTYyOCA2NDQuOTI1OTcsMTc0LjQxMDIxIDYz
Ny4wMDgzNSwxMzYuNzQxNTQgTCA3NTAuMDE0MzUsMjYwLjU3MjM0IHogIgogICAgIGlkPSJwYXRo
OTA2IgogICAgIHNvZGlwb2RpOm5vZGV0eXBlcz0iY2NjYyIKICAgICBzdHlsZT0iZmlsbDp1cmwo
I2xpbmVhckdyYWRpZW50MTQ5Nyk7ZmlsbC1vcGFjaXR5OjE7ZmlsbC1ydWxlOmV2ZW5vZGQ7c3Ry
b2tlLXdpZHRoOjAuOTU0MDc0OTVwdDsiCiAgICAgdHJhbnNmb3JtPSJtYXRyaXgoMC4yOTY2ODks
MC4wMDAwMDAsMC4wMDAwMDAsMC4yOTY2ODksLTcxLjE5NjAxLC0zNC4wNTgyNSkiIC8+CiAgPHBh
dGgKICAgICBkPSJNIDMwMS4zMDY1NSwxNzQuOTA4NjcgTCAyOTkuMjcwNjksNjI4LjkwNDM0IEwg
NzQzLjA4NzA5LDYyOC45MDQzNCBMIDc0My4zNDE1Nyw1MTYuNjc3ODMgQyA3MzMuNDE2NzcsNTIw
LjQ5NTA2IDYzMy4xNTA5MSwzNzguNDk0MTcgNTA2LjkyNzkwLDM4NC42MDE3MyBDIDM4MC41OTAy
OSwzOTAuNzE0ODQgNDEzLjI3ODU3LDE5My4yMzEzNiAzMDEuMzA2NTUsMTc0LjkwODY3IHogIgog
ICAgIGlkPSJwYXRoMTUwMCIKICAgICBzb2RpcG9kaTpub2RldHlwZXM9ImNjY2N6YyIKICAgICBz
dHlsZT0iZmlsbDp1cmwoI2xpbmVhckdyYWRpZW50MTQ5OSk7ZmlsbC1vcGFjaXR5OjE7ZmlsbC1y
dWxlOmV2ZW5vZGQ7c3Ryb2tlLXdpZHRoOjEuMDAwMDAwMHB0OyIKICAgICB0cmFuc2Zvcm09Im1h
dHJpeCgwLjI5NjY4OSwwLjAwMDAwMCwwLjAwMDAwMCwwLjI5NjY4OSwtNzEuMTk2MDEsLTM0LjA1
ODI1KSIgLz4KICA8dGV4dAogICAgIHhtbDpzcGFjZT0icHJlc2VydmUiCiAgICAgc3R5bGU9ImZv
bnQtc2l6ZTozNnB4O2ZvbnQtc3R5bGU6bm9ybWFsO2ZvbnQtdmFyaWFudDpub3JtYWw7Zm9udC13
ZWlnaHQ6bm9ybWFsO2ZvbnQtc3RyZXRjaDpub3JtYWw7dGV4dC1hbGlnbjpzdGFydDtsaW5lLWhl
aWdodDoxMjUlO3dyaXRpbmctbW9kZTpsci10Yjt0ZXh0LWFuY2hvcjpzdGFydDtmaWxsOiNkNDAw
MDA7ZmlsbC1vcGFjaXR5OjE7c3Ryb2tlOiMwMDAwMDA7c3Ryb2tlLXdpZHRoOjAuNTtzdHJva2Ut
bGluZWNhcDpidXR0O3N0cm9rZS1saW5lam9pbjptaXRlcjtzdHJva2Utb3BhY2l0eToxO2ZvbnQt
ZmFtaWx5OkNlbnR1cnkgU2Nob29sYm9vayBMOy1pbmtzY2FwZS1mb250LXNwZWNpZmljYXRpb246
Q2VudHVyeSBTY2hvb2xib29rIEw7c3Ryb2tlLW1pdGVybGltaXQ6NDtzdHJva2UtZGFzaGFycmF5
Om5vbmUiCiAgICAgeD0iNTUuMzA4NjQzIgogICAgIHk9IjEzNS4xNjc1NCIKICAgICBpZD0idGV4
dDI0MjkiCiAgICAgc29kaXBvZGk6bGluZXNwYWNpbmc9IjEyNSUiPjx0c3BhbgogICAgICAgc29k
aXBvZGk6cm9sZT0ibGluZSIKICAgICAgIGlkPSJ0c3BhbjI0MzEiCiAgICAgICB4PSI1NS4zMDg2
NDMiCiAgICAgICB5PSIxMzUuMTY3NTQiPlggPC90c3Bhbj48L3RleHQ+CiAgPHRleHQKICAgICB4
bWw6c3BhY2U9InByZXNlcnZlIgogICAgIHN0eWxlPSJmb250LXNpemU6MzZweDtmb250LXN0eWxl
Om5vcm1hbDtmb250LXdlaWdodDpub3JtYWw7ZmlsbDojMDA4MDAwO2ZpbGwtb3BhY2l0eToxO3N0
cm9rZTojMDAwMDAwO3N0cm9rZS13aWR0aDowLjU7c3Ryb2tlLWxpbmVjYXA6YnV0dDtzdHJva2Ut
bGluZWpvaW46bWl0ZXI7c3Ryb2tlLW9wYWNpdHk6MTtmb250LWZhbWlseTpDZW50dXJ5IFNjaG9v
bGJvb2sgTDstaW5rc2NhcGUtZm9udC1zcGVjaWZpY2F0aW9uOkNlbnR1cnkgU2Nob29sYm9vayBM
O2ZvbnQtc3RyZXRjaDpub3JtYWw7Zm9udC12YXJpYW50Om5vcm1hbDt0ZXh0LWFuY2hvcjpzdGFy
dDt0ZXh0LWFsaWduOnN0YXJ0O3dyaXRpbmctbW9kZTpscjtsaW5lLWhlaWdodDoxMjUlO3N0cm9r
ZS1taXRlcmxpbWl0OjQ7c3Ryb2tlLWRhc2hhcnJheTpub25lIgogICAgIHg9IjgxLjI2OTg0NCIK
ICAgICB5PSIxMzUuNDQ5NzIiCiAgICAgaWQ9InRleHQyNDMzIgogICAgIHNvZGlwb2RpOmxpbmVz
cGFjaW5nPSIxMjUlIj48dHNwYW4KICAgICAgIHNvZGlwb2RpOnJvbGU9ImxpbmUiCiAgICAgICBp
ZD0idHNwYW4yNDM1IgogICAgICAgeD0iODEuMjY5ODQ0IgogICAgICAgeT0iMTM1LjQ0OTcyIj5N
PC90c3Bhbj48L3RleHQ+CiAgPHRleHQKICAgICB4bWw6c3BhY2U9InByZXNlcnZlIgogICAgIHN0
eWxlPSJmb250LXNpemU6MzZweDtmb250LXN0eWxlOm5vcm1hbDtmb250LXZhcmlhbnQ6bm9ybWFs
O2ZvbnQtd2VpZ2h0Om5vcm1hbDtmb250LXN0cmV0Y2g6bm9ybWFsO3RleHQtYWxpZ246c3RhcnQ7
bGluZS1oZWlnaHQ6MTI1JTt3cml0aW5nLW1vZGU6bHItdGI7dGV4dC1hbmNob3I6c3RhcnQ7Zmls
bDojMDAwMDgwO2ZpbGwtb3BhY2l0eToxO3N0cm9rZTojMDAwMDAwO3N0cm9rZS13aWR0aDowLjU7
c3Ryb2tlLWxpbmVjYXA6YnV0dDtzdHJva2UtbGluZWpvaW46bWl0ZXI7c3Ryb2tlLW9wYWNpdHk6
MTtmb250LWZhbWlseTpDZW50dXJ5IFNjaG9vbGJvb2sgTDstaW5rc2NhcGUtZm9udC1zcGVjaWZp
Y2F0aW9uOkNlbnR1cnkgU2Nob29sYm9vayBMO3N0cm9rZS1taXRlcmxpbWl0OjQ7c3Ryb2tlLWRh
c2hhcnJheTpub25lIgogICAgIHg9IjExNC44NTAwOSIKICAgICB5PSIxMzUuNDQ5NzIiCiAgICAg
aWQ9InRleHQyNDM3IgogICAgIHNvZGlwb2RpOmxpbmVzcGFjaW5nPSIxMjUlIj48dHNwYW4KICAg
ICAgIHNvZGlwb2RpOnJvbGU9ImxpbmUiCiAgICAgICBpZD0idHNwYW4yNDM5IgogICAgICAgeD0i
MTE0Ljg1MDA5IgogICAgICAgeT0iMTM1LjQ0OTcyIgogICAgICAgc3R5bGU9ImZpbGwtb3BhY2l0
eToxO2ZpbGw6IzAwMDA4MDtzdHJva2U6IzAwMDAwMDtzdHJva2Utb3BhY2l0eToxO3N0cm9rZS13
aWR0aDowLjU7c3Ryb2tlLW1pdGVybGltaXQ6NDtzdHJva2UtZGFzaGFycmF5Om5vbmUiPkw8L3Rz
cGFuPjwvdGV4dD4KPC9zdmc+Cg=='''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'xml-pickle'
    self.name = 'XML Pickle'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'XML Persistence layer for complex objects'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.advertise ('xml.pickle', self.svc_xml_pickle)
    self.mediator.advertise ('xml.unpickle', self.svc_xml_unpickle)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: xml.pickle
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_xml_pickle (self, path, obj):
    pickle = Pickle ()
    pickle.mediator = self.mediator.copy ()
    pickle.save (path, obj)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: xml.unpickle
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_xml_unpickle (self, path):
    pickle = Pickle ()
    pickle.mediator = self.mediator.copy ()
    return pickle.load (path)
