/*
 * @(#)SymbolFactory.java
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * As a special exception, the copyright holders of this library 
 * give you permission to link this library with independent modules
 * to produce an executable, regardless of the license terms of 
 * these independent modules, and to copy and distribute the 
 * resulting executable under terms of your choice, provided that 
 * you also meet, for each linked independent module, the terms and 
 * conditions of the license of that module. An independent module 
 * is a module which is not derived from or based on this library. 
 * If you modify this library, you may extend this exception to your 
 * version of the library, but you are not obligated to do so. If 
 * you do not wish to do so, delete this exception statement from 
 * your version. 
 *
 * Copyright (c) 2003 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib;

import java.math.BigInteger;
import java.util.HashMap;

import net.percederberg.mib.symbol.Symbol;
import net.percederberg.mib.symbol.TypeSymbol;
import net.percederberg.mib.symbol.ValueSymbol;
import net.percederberg.mib.type.EnumerationType;
import net.percederberg.mib.type.IntegerType;
import net.percederberg.mib.type.NamedType;
import net.percederberg.mib.type.ObjectIdentifierType;
import net.percederberg.mib.type.StringType;
import net.percederberg.mib.type.Constraint;
import net.percederberg.mib.type.SizeConstraint;
import net.percederberg.mib.type.Type;
import net.percederberg.mib.type.ValueConstraint;
import net.percederberg.mib.type.ValueRangeConstraint;

/**
 * A factory class for creating symbols. This class is used to provide
 * access to the default symbols from both semantical analysis steps.
 * 
 * @author   Per Cederberg, per@percederberg.net
 * @version  1.1
 */
public class SymbolFactory {

    /**
     * The MIB to create the symbols for.
     */
    private Mib mib;

    /**
     * The symbols created in the factory. This table also contains 
     * all the intermediary symbols needed when creating requested 
     * symbols (such as nested OIDs).
     */
    private HashMap  symbols = new HashMap();

    /**
     * Creates a new symbol factory. The list of created symbols will 
     * initially be empty.
     * 
     * @param mib       the MIB where error should be reported
     */
    public SymbolFactory(Mib mib) {
        // TODO: move error reporting out of the Mib class
        this.mib = mib;
    }

    /**
     * Finds or creates a symbol with the specified known name. The 
     * symbol name must be one of the known SNMPv1 or SNMPv2 
     * definitions, or null will be returned. If a symbol with the
     * specified name has already been created by this factory, that 
     * symbol will be returned instead of creating a new symbol.
     *
     * @param  name    the symbol name (from the import)
     * 
     * @return the symbol found or created, or 
     *         null if the symbol name was unrecognized
     */
    public Symbol findSymbol(String name) {
        if (symbols.containsKey(name)) {
            return (Symbol) symbols.get(name);
        } else {
            return createSymbol(name);
        }
    }

    /**
     * Creates a symbol with the specified known name. The symbol name 
     * must be one of the known SNMPv1 or SNMPv2 definitions, or null 
     * will be returned.
     *
     * @param  name    the symbol name (from the import)
     * 
     * @return the symbol created, or 
     *         null if the symbol name was unrecognized
     */
    public Symbol createSymbol(String name) {
        Symbol  symbol = null;

        // Create symbol object
        if (symbol == null) {
            symbol = createObjectIdSymbol(name);
        }
        if (symbol == null) {
            symbol = createSnmp1TypeSymbol(name);
        }
        if (symbol == null) {
            symbol = createSnmp2TypeSymbol(name);
        }
        
        // Register symbol object
        if (symbol != null) {
            symbols.put(symbol.getName(), symbol);
        }

        return symbol;
    }

    /**
     * Creates an object id symbol with the specified known name. The 
     * symbol name must be one of the known SNMPv1 or SNMPv2 object 
     * id definitions, or null will be returned.
     *
     * @param  name    the symbol name (from the import)
     * 
     * @return the symbol created, or 
     *         null if the symbol name was unrecognized
     */
    private Symbol createObjectIdSymbol(String name) {
        Symbol  symbol;
        Type    type = ObjectIdentifierType.getInstance();

        if (name.equals("iso")) {
            return new ValueSymbol(name, type, 1);
        } else if (name.equals("org")) {
            symbol = findSymbol("iso");
            return new ValueSymbol(name, type, symbol, 3);
        } else if (name.equals("dod")) {
            symbol = findSymbol("org");
            return new ValueSymbol(name, type, symbol, 6);
        } else if (name.equals("internet")) {
            symbol = findSymbol("dod");
            return new ValueSymbol(name, type, symbol, 1);
        } else if (name.equals("directory")) {
            symbol = findSymbol("internet");
            return new ValueSymbol(name, type, symbol, 1);
        } else if (name.equals("mgmt")) {
            symbol = findSymbol("internet");
            return new ValueSymbol(name, type, symbol, 2);
        } else if (name.equals("mib-2")) {
            symbol = findSymbol("mgmt");
            return new ValueSymbol(name, type, symbol, 1);
        } else if (name.equals("transmission")) {
            symbol = findSymbol("mib-2");
            return new ValueSymbol(name, type, symbol, 10);
        } else if (name.equals("experimental")) {
            symbol = findSymbol("internet");
            return new ValueSymbol(name, type, symbol, 3);
        } else if (name.equals("private")) {
            symbol = findSymbol("internet");
            return new ValueSymbol(name, type, symbol, 4);
        } else if (name.equals("enterprises")) {
            symbol = findSymbol("private");
            return new ValueSymbol(name, type, symbol, 1);
        } else if (name.equals("security")) {
            symbol = findSymbol("internet");
            return new ValueSymbol(name, type, symbol, 5);
        } else if (name.equals("snmpV2")) {
            symbol = findSymbol("internet");
            return new ValueSymbol(name, type, symbol, 6);
        } else if (name.equals("snmpDomains")) {
            symbol = findSymbol("snmpV2");
            return new ValueSymbol(name, type, symbol, 1);
        } else if (name.equals("snmpProxys")) {
            symbol = findSymbol("snmpV2");
            return new ValueSymbol(name, type, symbol, 2);
        } else if (name.equals("snmpModules")) {
            symbol = findSymbol("snmpV2");
            return new ValueSymbol(name, type, symbol, 3);
        } else if (name.equals("zeroDotZero")) {
            symbol = new ValueSymbol("zero", type, 0);
            return new ValueSymbol(name, type, symbol, 0);
        } else {
            return null;
        }
    }

    /**
     * Creates a type symbol with the specified known name. The symbol 
     * name must be one of the known SNMPv1 type definitions, or null 
     * will be returned.
     *
     * @param  name    the symbol name (from the import)
     * 
     * @return the symbol created, or 
     *         null if the symbol name was unrecognized
     */
    private Symbol createSnmp1TypeSymbol(String name) {
        Symbol     symbol = new TypeSymbol(name);
        Type       type = null;
        Constraint cons = null;

        if (name.equals("ObjectName")) {
            type = ObjectIdentifierType.getInstance();
            symbol.setType(type);
        } else if (name.equals("ObjectSyntax")) {
            mib.addWarning("choice types are not supported: " + name);
        } else if (name.equals("SimpleSyntax")) {
            mib.addWarning("choice types are not supported: " + name);
        } else if (name.equals("ApplicationSyntax")) {
            mib.addWarning("choice types are not supported: " + name);
        } else if (name.equals("NetworkAddress")) {
            mib.addWarning("choice types are not supported: " + name);
        } else if (name.equals("IpAddress")) {
            cons = new ValueConstraint(new Integer(4));
            cons = new SizeConstraint((ValueConstraint) cons);
            // TODO: Create a new octet string type
            symbol.setType(new StringType((SizeConstraint) cons));
        } else if (name.equals("Counter")) {
            cons = new ValueRangeConstraint(new Long(0),
                                            new Long(4294967295L));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("Gauge")) {
            cons = new ValueRangeConstraint(new Long(0),
                                            new Long(4294967295L));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("TimeTicks")) {
            cons = new ValueRangeConstraint(new Long(0),
                                            new Long(4294967295L));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("Opaque")) {
            // TODO: Create a new octet string type
            symbol.setType(new StringType());
        } else {
            return null;
        }
        return symbol;
    }

    /**
     * Creates a type symbol with the specified known name. The symbol 
     * name must be one of the known SNMPv2 type definitions, or null 
     * will be returned.
     *
     * @param  name    the symbol name (from the import)
     * 
     * @return the symbol created, or 
     *         null if the symbol name was unrecognized
     */
    private Symbol createSnmp2TypeSymbol(String name) {
        Symbol           symbol = new TypeSymbol(name);
        Type             type = null;
        EnumerationType  enum = null;
        Constraint       cons = null;

        if (name.equals("ObjectName")) {
            type = ObjectIdentifierType.getInstance();
            symbol.setType(type);
        } else if (name.equals("NotificationName")) {
            type = ObjectIdentifierType.getInstance();
            symbol.setType(type);
        } else if (name.equals("ObjectSyntax")) {
            mib.addWarning("choice types are not supported: " + name);
        } else if (name.equals("SimpleSyntax")) {
            mib.addWarning("choice types are not supported: " + name);
        } else if (name.equals("Integer32")) {
            cons = new ValueRangeConstraint(new Long(-2147483648L),
                                            new Long(2147483647L));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("ApplicationSyntax")) {
            mib.addWarning("choice types are not supported: " + name);
        } else if (name.equals("IpAddress")) {
            cons = new ValueConstraint(new Integer(4));
            cons = new SizeConstraint((ValueConstraint) cons);
            // TODO: Create a new octet string type
            symbol.setType(new StringType((SizeConstraint) cons));
        } else if (name.equals("Counter32")) {
            cons = new ValueRangeConstraint(new Long(0),
                                            new Long(4294967295L));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("Gauge32")) {
            cons = new ValueRangeConstraint(new Long(0),
                                            new Long(4294967295L));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("Unsigned32")) {
            cons = new ValueRangeConstraint(new Long(0),
                                            new Long(4294967295L));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("TimeTicks")) {
            cons = new ValueRangeConstraint(new Long(0),
                                            new Long(4294967295L));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("Opaque")) {
            // TODO: Create a new octet string type
            symbol.setType(new StringType());
        } else if (name.equals("Counter64")) {
            cons = new ValueRangeConstraint(BigInteger.ZERO,
                                            new BigInteger("18446744073709551615"));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("DisplayString")) {
            cons = new ValueRangeConstraint(new Integer(0), 
                                            new Integer(255));
            cons = new SizeConstraint((ValueRangeConstraint) cons);
            symbol.setType(new StringType((SizeConstraint) cons));
        } else if (name.equals("PhysAddress")) {
            cons = new ValueConstraint(new Integer(6));
            cons = new SizeConstraint((ValueConstraint) cons);
            // TODO: Create a new octet string type
            symbol.setType(new StringType((SizeConstraint) cons));
        } else if (name.equals("TruthValue")) {
            enum = new EnumerationType();
            enum.addValue("true", new Integer(1));
            enum.addValue("false", new Integer(2));
            symbol.setType(enum);
        } else if (name.equals("TestAndIncr")) {
            cons = new ValueRangeConstraint(new Long(0),
                                            new Long(2147483647L));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("AutonomousType")) {
            symbol.setType(ObjectIdentifierType.getInstance());
        } else if (name.equals("InstancePointer")) {
            symbol.setType(ObjectIdentifierType.getInstance());
        } else if (name.equals("VariablePointer")) {
            symbol.setType(ObjectIdentifierType.getInstance());
        } else if (name.equals("RowPointer")) {
            symbol.setType(ObjectIdentifierType.getInstance());
        } else if (name.equals("RowStatus")) {
            enum = new EnumerationType();
            enum.addValue("active", new Integer(1));
            enum.addValue("notInService", new Integer(2));
            enum.addValue("notReady", new Integer(3));
            enum.addValue("createAndGo", new Integer(4));
            enum.addValue("createAndWait", new Integer(5));
            enum.addValue("destroy", new Integer(6));
            symbol.setType(enum);
        } else if (name.equals("TimeStamp")) {
            type = new NamedType((TypeSymbol) findSymbol("TimeTicks"));
            symbol.setType(type);
        } else if (name.equals("TimeInterval")) {
            cons = new ValueRangeConstraint(new Long(0),
                                            new Long(2147483647L));
            symbol.setType(new IntegerType((ValueRangeConstraint) cons));
        } else if (name.equals("DateAndTime")) {
            // TODO: Create a size constraint for only 8 and 11 
            //       (nothing in between).
            cons = new ValueRangeConstraint(new Integer(8), 
                                            new Integer(11));
            cons = new SizeConstraint((ValueRangeConstraint) cons);
            // TODO: Create a new octet string type
            symbol.setType(new StringType((SizeConstraint) cons));
        } else if (name.equals("StorageType")) {
            enum = new EnumerationType();
            enum.addValue("other", new Integer(1));
            enum.addValue("volatile", new Integer(2));
            enum.addValue("nonVolatile", new Integer(3));
            enum.addValue("permanent", new Integer(4));
            enum.addValue("readOnly", new Integer(5));
            symbol.setType(enum);
        } else if (name.equals("TDomain")) {
            symbol.setType(ObjectIdentifierType.getInstance());
        } else if (name.equals("TAddress ")) {
            cons = new ValueRangeConstraint(new Integer(1), 
                                            new Integer(255));
            cons = new SizeConstraint((ValueRangeConstraint) cons);
            // TODO: Create a new octet string type
            symbol.setType(new StringType((SizeConstraint) cons));
        } else if (name.equals("SnmpAdminString")) {
            cons = new ValueRangeConstraint(new Integer(0), 
                                            new Integer(255));
            cons = new SizeConstraint((ValueRangeConstraint) cons);
            symbol.setType(new StringType((SizeConstraint) cons));
        } else {
            return null;
        }
        return symbol;
    }


}
