/*
	This file is part of jrisk.
	
	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 14-feb-2005 by Alessio Treglia
 * Copyright  2005 Alessio Treglia
 */
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collections;
/**
 * Modella un giocatore umano. Possiede come caratteristiche
 * una stringa contenente il nome del colore delle armate del
 * giocatore, una collezione di carte e un elenco dei territori
 * occupati dal giocatore. Inoltre tiene in memoria un riferimento
 * ad un oggetto Display, usato per la visualizzazione dei
 * messaggi del gioco.
 * 
 * @author Alessio Treglia
 * @version 1.0.0
 */
public class Giocatore {
    /** Colore delle armate del giocatore*/
    private String colore;
    /** Carte possedute dal giocatore */
    private List carte;    
    /** Territori occupati dalle truppe del giocatore */
    private Set territori;
    /** Indica se il giocatore ha conquistato almeno un
     * territorio durante il suo turno */
    private boolean territorioVinto;
    /** Schermo del giocatore */
    public Display schermo;
    /** Interpete dei comandi */
    public Parser parser;
    /** Turno corrente */
    private static int turno = 0;
    
    /**
     * All'inizio del gioco ogni giocatore sceglie un colore, dunque
     * gli vengono assegnati n territori a caso.
     * 
     * @param colore Colore delle truppe del giocatore
     */
    public Giocatore(String colore) {
        this.colore = colore;
        //this.territori.addAll(territori);
        this.territori = new HashSet();
        this.schermo = new Display();
        this.parser = new Parser();
        this.carte = new ArrayList();
        this.territorioVinto = false;
    }
    
    /**
     * Assegna una periferica di visualizzazione al Giocatore.
     * 
     * @param schermo periferica di visualizzazione usata dal
     * gioco per comunicare con il giocatore.
     */
    public void setDisplay(Display schermo) {
        this.schermo = schermo;
    }
    
    /**
     * Restitusice il colore delle armate del giocatore
     * 
     * @return il colore delle armate del giocatore
     */
    public String getColore() {
        return this.colore;
    }    
    /**
     * Restituisce una collezione di tipo <code>Set</code>
     * contenente tutte le carte possedute dal giocatore.
     * 
     * @return le carte possedute dal giocatore
     */
    public List getCarte() { return this.carte; }
    /**
     * Aggiunge una carta fra quelle possedute dal giocatore.
     * 
     * @param valore carta da aggiungere nella collezione di carte
     * possedute dal giocatore
     */
    public void aggiungiCarta(String valore) {
        if(ElencoCarte.cartaValida(valore))
            carte.add(valore);
        this.territorioVinto = false;
    }
    /**
     * Aggiunge una serie di carte a quelle gi
     * possedute. Utile in caso di sconfitta di un
     * giocatore, poich il giocatore sconfitto cede
     * le carte al giocatore che lo ha eliminato.
     * 
     * @param carte <code>List</code> di carte da aggiungere
     */
    public void aggiungiCarte(List carte) {
        this.carte.addAll(carte);
    }
    /**
     * Rimuove una carta di un dato valore dalla
     * collezione di carte possedute dal giocatore
     * 
     * @param valore valore della carta da rimuovere
     * dalla collezione di carte
     * @return nome <code>String</code> della carta
     */
    public String rimuoviCarta(String valore) {
        Iterator it = carte.iterator();
        String temp;
        while(it.hasNext()) {
            temp = (String) it.next();
            if(temp.equals("valore")) {
                it.remove();
                return temp;
            }
        }
        /*if(carte.contains(valore)) {
            carte.remove(valore);
            return valore;
        }*/
        return null;            
    }
    /**
     * Rimuove tutte le carte dalla collezione
     * di carte possedute dal giocatore. Utile
     * quando un giocatore  stato sconfitto,
     * poich scarica le carte dalla memoria
     * e le passa a un altro oggetto giocatore.
     * 
     * @return tutte le carte del giocatore, dopo
     * averle rimosse
     */
    public List rimuoviTutteLeCarte() {
        Iterator it = carte.iterator();
        List rimosse = new ArrayList();
        String carta;
        
        while(it.hasNext()) {
            carta = (String) it.next();
            rimosse.add(carta);
            it.remove();
        }
        return rimosse;
    }
    
    /**
     * Restituisce il numero di carte possedute dal giocatore
     * 
     * @return il numero di carte possedute dal giocatore
     */
    public int numeroCarte() { return carte.size(); }
    /**
     * Rimuove n carte uguali dalla collezione di carte possedute
     * dal giocatore e le restituisce in una collezione.
     * 
     * @param valore valore <code>String</code> delle carte da rimuovere
     * @param n numero di carte uguali da rimuovere
     * @return una <code>List</code> contenente le carte rimosse
     */
    public List rimuoviCarteStessoValore(String valore, int n) {
        int occorrenzeValore = Collections.frequency(this.carte, valore);
        if(n <= occorrenzeValore) {
            List carteRimosse = new ArrayList();
            for(int i=0; i<n; i++) {
                carteRimosse.add(valore);
                carte.remove(valore);
            }
        }
        
        return null;
    }
    
    /**
     * Verifica la presenza di un territorio nella collezione
     * dei territori posseduti dal giocatore, e lo restituisce.
     * 
     * @param nomeTerritorio nome del territorio
     * @return il riferimento al <code>Territorio</code>, se effettivamente
     * posseduto, altrimenti <code>null</code>
     */
    public Territorio getTerritorio(String nomeTerritorio) {
        Iterator it = territori.iterator();
        Territorio temp;
        
        while(it.hasNext()) {
            temp = (Territorio) it.next();
            if(temp.getNome().equals(nomeTerritorio))
                return temp;
        }
        return null;
    }
    
    /**
     * Restituisce una collezione di tipo Set contenente
     * tutti i territori posseduti dal giocatore.
     * 
     * @return i territori posseduti dal giocatore
     */
    public Set getTerritori() {
        return this.territori;
    }
    
    /**
     * Quando viene perso un territorio in battaglia,
     * questo viene eliminato dalla collezione di territori
     * posseduti.
     * 
     * @param nomeTerritorio nome del territorio perso
     * @return il <code>Territorio</code> perso
     */
    public Territorio rimuoviTerritorio(String nomeTerritorio) {
        Iterator it = territori.iterator();
        Territorio temp;
        while(it.hasNext()) {
            temp = (Territorio) it.next();
            if(temp.getNome().equals(nomeTerritorio)) {
                it.remove();
                return temp;
            }  
        }
        return null;
    }
    
    /**
     * In caso di vittoria, il territorio vinto viene aggiunto
     * fra quelli posseduti.
     * 
     * @param t <code>Territorio</code> vinto
     */
    public void aggiungiTerritorio(Territorio t) {
        territori.add(t);
    }
    
    /**
     * Comunica al giocatore se ha la possibilit di pescare
     * una carta Territori al prossimo giro.
     * 
     * @param haVintoUnTerritorio <code>true</code> se il
     * giocatore ha conquistato almeno un <code>Territorio</code>
     */
    public void setPuoPescare(boolean haVintoUnTerritorio) {
        this.territorioVinto = haVintoUnTerritorio;
    }
    /**
     * Verifica se il giocatore pu pescare una carta Territori
     * 
     * @return <code>true</code> se nel turno scorso ha conquistato almeno
     * un territorio
     */
    public boolean getPuoPescare() { return this.territorioVinto; }
    /**
     * Restituisce il turno corrente del giocatore
     * 
     * @return turni giocati dal giocatore
     */
    public static int getTurnoCorrente() { return turno; }
    /**
     * Aggiorna i numeri di turni giocati dal <code>Giocatore</code>
     *
     * return <code>void</code>
     */
    public static void aggiornaTurni() { turno += 1; }
    /**
     * Controlla se il giocatore  ancora in gioco.
     * 
     * @return <code>true</code> se il <code>Giocatore</code>
     * occupa almeno un <code>Territorio</code>
     */
    public boolean eliminato() {
        if(territori.size()==0)
            return true;
        return false;
    }
    /**
     * Verifica la condizione di vittoria.
     * 
     * @return <code>true</code> se il giocatore ha vinto,
     * cio se occupa tutti i territori del <code>Tabellone</code>
     */
    public boolean haVinto() {
        if(territori.size() == ElencoTerritoriContinenti.TERRITORI.length)
            return true;
        return false;
    }
    /**
     * Restituisce una descrizione stringa dell'oggetto.
     * 
     * @return descrizione <code>String</code> dell'oggetto
     */
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        Iterator it = territori.iterator();
        Territorio temp;
        
        buffer.append(colore + " " + territori.size() + " territori: ");
        while(it.hasNext()) {
            temp = (Territorio) it.next();
            buffer.append("\n\t" + temp.getNome() + " " +
                    temp.getArmate() + " armate");
        }
        
        return buffer.toString();
    }
}
