/* Copyright (c) 2012 - 2013 Samuel Klipphahn
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

   * Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   * Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   * Neither the name of the authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. */

/* $Id: $ */
/**
 * @file
 * @brief MAC functions of the Simple Routing Protocoll.
 * @_addtogroup grplwMesh
 */


/* === includes ============================================================ */
#include "lw_mesh.h"
#include "lw_route.h"
#include "board.h"
#include "radio.h"
#include "transceiver.h"
#include <string.h>

/* === macros ============================================================== */

/* === types =============================================================== */

/* === globals ============================================================= */
static uint16_t m_short_addr;
static uint16_t m_pan_id;
static uint8_t  m_seq_nr;
static uint8_t  last_task_id;

/* === prototypes ========================================================== */

/* === functions =========================================================== */

void lwm_init(uint16_t pan_id, uint16_t short_addr, uint8_t channel)
{
    /* init global variables */
    m_pan_id = pan_id;
    m_short_addr = short_addr;
    m_seq_nr = 0;
    last_task_id = NWK_NO_TASK_ID;

    /* buffer for next frame */
    uint8_t *rxFrame = malloc(sizeof(NWK_FrameFormat_t));
    radio_init(rxFrame, sizeof(NWK_FrameFormat_t));
    radio_set_param(RP_PANID(pan_id));
    radio_set_param(RP_SHORTADDR(short_addr));
    radio_set_param(RP_CHANNEL(channel));

#if defined(USE_RX_AACK)
    radio_set_param(RP_IDLESTATE(STATE_RXAUTO));
#elif defined(NWK_SLEEPING_NODE)
    radio_set_param(RP_IDLESTATE(STATE_SLEEP));
#else
    radio_set_param(RP_IDLESTATE(STATE_RX));
#endif
}

radio_cca_t lwm_send_frame(uint8_t payload_size, NWK_FrameFormat_t *frame,
        uint8_t task_id, uint8_t options)
{
    last_task_id = task_id;
    if(frame->m_dstAddr == NWK_BROADCAST_ADDR)
    {
        frame->m_fctl = NWK_MAC_FCTL_BROADCAST;
    } else {
        frame->m_fctl = NWK_MAC_FCTL_UNICAST;
    }
    frame->m_seq        = m_seq_nr++;
    if(options & NWK_OPT_BROADCAST_PAN_ID)
    {
        frame->m_pid    = NWK_BROADCAST_PID;
    } else {
        frame->m_pid    = m_pan_id;
    }
    frame->m_srcAddr    = m_short_addr;

    radio_cca_t cca;
#if defined(USE_RX_AACK)
    cca = RADIO_CCA_FREE; /* assume nothing went wrong during CCA, in other
    cases we get informed by usr_radio_tx_done callback */
    radio_set_state(STATE_TXAUTO);
    radio_send_frame(NWK_HEADER_SIZE + payload_size, (uint8_t *) frame, 0);
#else
    cca = radio_do_cca();
    if(cca == RADIO_CCA_FREE)
    {
        radio_set_state(STATE_TX);
        radio_send_frame(NWK_HEADER_SIZE + payload_size, (uint8_t *) frame, 0);
    }
#endif
    return cca;
}

/* -- radio interface functions -- */

uint8_t * usr_radio_receive_frame(uint8_t len, uint8_t *frm, uint8_t lqi,
 int8_t rssi, uint8_t crc_fail)
{
    uint8_t *nextFrame;
    if(!crc_fail)
    {
        lw_data_indication((NWK_FrameFormat_t *) frm, len - NWK_HEADER_SIZE,
                lqi, rssi);
        nextFrame = malloc(sizeof(NWK_FrameFormat_t));
    } else {
        nextFrame = frm;
    }
    return nextFrame;
}

void usr_radio_tx_done(radio_tx_done_t status)
{
    lw_tx_done(last_task_id, status);

    /* radio sets idle state
     *
     * TODO: if sleeping node and NWK Ack should be used, we have to switch to Rx and
     * start timer to wait for Ack frame
     */
}
