/* $Id: TspAggrTest.java,v 1.4 2005/10/05 00:11:47 sgalles Exp $
 * -----------------------------------------------------------------------
 * 
 * TSP Library - core components for a generic Transport Sampling Protocol.
 * 
 * Copyright (c) 2002 Yves DUFRENNE, Stephane GALLES, Eric NOULARD and Robert PAGNOT 
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * -----------------------------------------------------------------------
 * 
 * Project    : TSP
 * SubProject : jTSP
 * Maintainer : tsp@astrium-space.com
 * Component  : Consumer
 * 
 * -----------------------------------------------------------------------
 * 
 * Purpose   : 
 * 
 * -----------------------------------------------------------------------
 */
package tsp.consumer.util.aggregator;


import tsp.core.consumer.simple.TspSimpleConsumer;
import tsp.core.consumer.simple.TspSimpleRequestSample;
import tsp.core.consumer.simple.TspSimpleSample;
import tsp.core.consumer.simple.TspSimpleSampleSymbolInfo;
import tsp.core.consumer.simple.TspSimpleSession;
import junit.framework.TestCase;

public class TspAggrTest extends TestCase {
	
	private TspSimpleConsumer aggrConsumer;
	
	public final static TspSimpleSampleSymbolInfo[] SAMPLE_HOST1 = {
			new TspSimpleSampleSymbolInfo("n1s1", 1, 1, 0),
			new TspSimpleSampleSymbolInfo("n1s2", 1, 1, 1),
			new TspSimpleSampleSymbolInfo("n1s3", 1, 1, 2),
	};
	public final static TspSimpleSampleSymbolInfo[] SAMPLE_HOST2 = {
			new TspSimpleSampleSymbolInfo("n2s1", 1, 1, 0),			
	};
	public final static TspSimpleSampleSymbolInfo[] SAMPLE_HOST3 = {
			new TspSimpleSampleSymbolInfo("n3s1", 1, 1, 0),
			new TspSimpleSampleSymbolInfo("n3s2", 1, 1, 1),			
	};
	public final static TspSimpleSampleSymbolInfo[] SAMPLE_HOST4 = {
			new TspSimpleSampleSymbolInfo("n4s1", 1, 1, 0),
			new TspSimpleSampleSymbolInfo("n4s2", 1, 1, 1),
			new TspSimpleSampleSymbolInfo("n4s3", 1, 1, 2),
			new TspSimpleSampleSymbolInfo("n4s4", 1, 1, 3),
	};
	
	
	
	public static TspAggrNamedConsumer[] consumers = {
			new TspAggrNamedConsumer("na1", new MockSimpleConsumer(SAMPLE_HOST1)),
			new TspAggrNamedConsumer("na2", new MockSimpleConsumer(SAMPLE_HOST2)),
			new TspAggrNamedConsumer("na3", new MockSimpleConsumer(SAMPLE_HOST3)),
			new TspAggrNamedConsumer("na4", new MockSimpleConsumer(SAMPLE_HOST4)),
	};
	
	public static TspSimpleSampleSymbolInfo[] REQ_INFOS={
			new TspSimpleSampleSymbolInfo("na4::n4s2", 2, 6),
			new TspSimpleSampleSymbolInfo("na4::n4s4", 3, 8),
			new TspSimpleSampleSymbolInfo("na1::n1s2", 3, 8),
			new TspSimpleSampleSymbolInfo("na2::n2s1", 1, 2),
	};
	
	
	
	public void setUp() throws Exception{
			
		aggrConsumer = new TspAggrConsumerImpl(consumers);
	}
	
	public void testRequestInfo() throws Exception{		
		TspSimpleSession session = aggrConsumer.openSession();
		TspSimpleSampleSymbolInfo[] infos = session.requestInfos().getSymbolInfos();
		assertEquals("na1::n1s1", infos[0].getName());
		assertEquals(1, infos[0].getPeriod());
		assertEquals(1, infos[0].getPhase());
		assertEquals("na4::n4s4", infos[9].getName());
	}
	
	public void testRequestSample() throws Exception{		
		TspSimpleSession session = aggrConsumer.openSession();		
				
		// try the request
		TspSimpleSampleSymbolInfo[] infos = session.requestSample(new TspSimpleRequestSample(REQ_INFOS)).getSymbolInfos();
		assertEquals(4, infos.length);
		assertEquals("na4::n4s2", infos[0].getName());
		assertEquals(0, infos[0].getIndex());
		assertEquals(2, infos[0].getPeriod());
		assertEquals(6, infos[0].getPhase());
		assertEquals("na4::n4s4", infos[1].getName());
		assertEquals(1, infos[1].getIndex());
		assertEquals("na1::n1s2", infos[2].getName());
		assertEquals(2, infos[2].getIndex());
		assertEquals("na2::n2s1", infos[3].getName());
		assertEquals(3, infos[3].getIndex());
		
		// check that requestSample was not called for the session
		// for wich samples are not used
		MockSimpleConsumer mock = null;
		mock = (MockSimpleConsumer)consumers[0].getConsumer();
		assertTrue(mock.getSession().isRequestSampleCalled());
		mock = (MockSimpleConsumer)consumers[1].getConsumer();
		assertTrue(mock.getSession().isRequestSampleCalled());
		// na3 not used
		mock = (MockSimpleConsumer)consumers[2].getConsumer();
		assertFalse(mock.getSession().isRequestSampleCalled());
		mock = (MockSimpleConsumer)consumers[3].getConsumer();
		assertTrue(mock.getSession().isRequestSampleCalled());
		
	}
	
	public void testRequestSampleInit() throws Exception{		
		TspSimpleSession session = aggrConsumer.openSession();		
				
		// try the request
		session.requestSample(new TspSimpleRequestSample(REQ_INFOS));
		
		session.requestSampleInit();
		// check that requestSampleInit was not called for the session
		// for wich samples are not used
		MockSimpleConsumer mock = null;
		mock = (MockSimpleConsumer)consumers[0].getConsumer();
		assertTrue(mock.getSession().isRequestSampleInitCalled());
		mock = (MockSimpleConsumer)consumers[1].getConsumer();
		assertTrue(mock.getSession().isRequestSampleInitCalled());
		// na3 not used
		mock = (MockSimpleConsumer)consumers[2].getConsumer();
		assertFalse(mock.getSession().isRequestSampleInitCalled());
		mock = (MockSimpleConsumer)consumers[3].getConsumer();
		assertTrue(mock.getSession().isRequestSampleInitCalled());		
		
	}
	
		
	
	public void testIsDataAvailable() throws Exception{		
		TspSimpleSession session = aggrConsumer.openSession();		
				
		// try the request
		session.requestSample(new TspSimpleRequestSample(REQ_INFOS));
		
		session.requestSampleInit();
				
		// check that requestSampleInit was not called for the session
		// for wich samples are not used		
		MockSimpleConsumer mock1 = (MockSimpleConsumer)consumers[0].getConsumer();
		MockSimpleConsumer mock2 = (MockSimpleConsumer)consumers[1].getConsumer();
		MockSimpleConsumer mock3 = (MockSimpleConsumer)consumers[2].getConsumer();
		MockSimpleConsumer mock4 = (MockSimpleConsumer)consumers[3].getConsumer();
		
		// not data;
		assertFalse(session.isDataAvailable());
		
		// data in the unused session (can not happen, but...)
		mock3.getSession().addSample(new TspSimpleSample(1,2));
		assertFalse(session.isDataAvailable());
		mock3.getSession().clearAllSamples();
		
		// data in one session
		mock2.getSession().addSample(new TspSimpleSample(1,2));
		assertTrue(session.isDataAvailable());
		
		// data all used sessions
		mock1.getSession().addSample(new TspSimpleSample(1,2));
		mock4.getSession().addSample(new TspSimpleSample(1,2));
		assertTrue(session.isDataAvailable());
		
	}
	
	public void testFetchSample() throws Exception{		
		TspSimpleSession session = aggrConsumer.openSession();		
				
		// try the request
		session.requestSample(new TspSimpleRequestSample(REQ_INFOS));		
		session.requestSampleInit();
				
		MockSimpleConsumer mock1 = (MockSimpleConsumer)consumers[0].getConsumer();
		MockSimpleConsumer mock2 = (MockSimpleConsumer)consumers[1].getConsumer();
		MockSimpleConsumer mock3 = (MockSimpleConsumer)consumers[2].getConsumer();
		MockSimpleConsumer mock4 = (MockSimpleConsumer)consumers[3].getConsumer();
		
		// add data 
		// external indice : 2
		mock1.getSession().addSample(new TspSimpleSample(1, 3.1));
		mock1.getSession().addSample(new TspSimpleSample(1, 3.2));
		mock1.getSession().addSample(new TspSimpleSample(1, 3.3));
		// external indice : 3
		mock2.getSession().addSample(new TspSimpleSample(0, 10.1));
		mock2.getSession().addSample(new TspSimpleSample(0, 10.2));
		// external indice : 1
		mock4.getSession().addSample(new TspSimpleSample(3, 100.1));
		mock4.getSession().addSample(new TspSimpleSample(3, 100.2));
		//external indice : 0 
		mock4.getSession().addSample(new TspSimpleSample(1, 11.1));
		mock4.getSession().addSample(new TspSimpleSample(1, 11.2));
		
		TspSimpleSample s = new TspSimpleSample();
		
		// external indice : 2
		session.fetchSample(s);		
		assertEquals(2, s.getIndex());
		assertEquals(3.1, s.getValue(), 0);
		session.fetchSample(s);		
		assertEquals(2, s.getIndex());
		assertEquals(3.2, s.getValue(), 0);
		session.fetchSample(s);		
		assertEquals(2, s.getIndex());
		assertEquals(3.3, s.getValue(), 0);
//		 external indice : 3
		session.fetchSample(s);		
		assertEquals(3, s.getIndex());
		assertEquals(10.1, s.getValue(), 0);
		session.fetchSample(s);		
		assertEquals(3, s.getIndex());
		assertEquals(10.2, s.getValue(), 0);
//		 external indice : 1
		session.fetchSample(s);
		assertEquals(1, s.getIndex());
		assertEquals(100.1, s.getValue(), 0);
		session.fetchSample(s);		
		assertEquals(1, s.getIndex());
		assertEquals(100.2, s.getValue(), 0);
//		external indice : 0
		session.fetchSample(s);				
		assertEquals(0, s.getIndex());
		assertEquals(11.1, s.getValue(), 0);
		session.fetchSample(s);		
		assertEquals(0, s.getIndex());
		assertEquals(11.2, s.getValue(), 0);
		
	}
	
	public void testRequestSampleFinalize() throws Exception{		
		TspSimpleSession session = aggrConsumer.openSession();		
				
		// try the request
		session.requestSample(new TspSimpleRequestSample(REQ_INFOS));		
		session.requestSampleInit();
		
		// call...
		session.requestSampleFinalize();
		// check that requestSampleFinalize was not called for the session
		// for wich samples are not used
		MockSimpleConsumer mock = null;
		mock = (MockSimpleConsumer)consumers[0].getConsumer();
		assertTrue(mock.getSession().isRequestSampleFinalizeCalled());
		mock = (MockSimpleConsumer)consumers[1].getConsumer();
		assertTrue(mock.getSession().isRequestSampleFinalizeCalled());
		// na3 not used
		mock = (MockSimpleConsumer)consumers[2].getConsumer();
		assertFalse(mock.getSession().isRequestSampleFinalizeCalled());
		mock = (MockSimpleConsumer)consumers[3].getConsumer();
		assertTrue(mock.getSession().isRequestSampleFinalizeCalled());		
		
	}

	
	public void testClose() throws Exception{		
		TspSimpleSession session = aggrConsumer.openSession();		
				
		// try the request
		session.requestSample(new TspSimpleRequestSample(REQ_INFOS));		
		session.requestSampleInit();
		session.requestSampleFinalize();
		
		// call...		
		session.close();
		
		MockSimpleConsumer mock = null;
		mock = (MockSimpleConsumer)consumers[0].getConsumer();
		assertTrue(mock.getSession().isCloseCalled());
		mock = (MockSimpleConsumer)consumers[1].getConsumer();
		assertTrue(mock.getSession().isCloseCalled());		
		mock = (MockSimpleConsumer)consumers[2].getConsumer();
		assertTrue(mock.getSession().isCloseCalled());
		mock = (MockSimpleConsumer)consumers[3].getConsumer();
		assertTrue(mock.getSession().isCloseCalled());		
		
	}

	
}
