// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief C++ API module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "registry.h"
#include "registry_key.h"
#include "registry_data.h"
#include "registry_file.h"
#include "registry_value.h"
#include "api.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create registry from C++ object
//! \param r C++ registry object
//! \return new registry object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
PyMobius_Forensics_Registry_Registry_from_cpp (mobius::forensics::registry::registry r)
{
  PyObject *ret = _PyObject_New (&forensics_registry_registry_t);

  if (ret)
    ((forensics_registry_registry_o *) ret)->obj = new mobius::forensics::registry::registry (r);

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: tp_new (default constructor)
//! \param type type object
//! \param args argument list
//! \param kwds keywords dict
//! \return new registry object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_new (PyTypeObject *type, PyObject *, PyObject *)
{
  forensics_registry_registry_o *self = (forensics_registry_registry_o *) type->tp_alloc (type, 0);

  if (self)
    self->obj = new mobius::forensics::registry::registry ();

  return (PyObject *) self;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
tp_dealloc (forensics_registry_registry_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: files getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_files (forensics_registry_registry_o *self)
{
  PyObject *ret = PyList_New (0);
  if (!ret)
    return nullptr;

  try
    {
      for (auto f : self->obj->get_files ())
        {
          PyObject *item = PyMobius_Forensics_Registry_Registry_File_from_cpp (f);

          if (!item)
            {
              Py_CLEAR (ret);
              return nullptr;
            }

          PyList_Append (ret, item);
          Py_DECREF (item);
        }
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      Py_CLEAR (ret);
      ret = nullptr;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: keys getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_keys (forensics_registry_registry_o *self)
{
  PyObject *ret = PyList_New (0);
  if (!ret)
    return nullptr;

  try
    {
      for (auto k : self->obj->get_keys ())
        {
          PyObject *item = PyMobius_Forensics_Registry_Registry_Key_from_cpp (k);

          if (!item)
            {
              Py_CLEAR (ret);
              return nullptr;
            }

          PyList_Append (ret, item);
          Py_DECREF (item);
        }
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      Py_CLEAR (ret);
      ret = nullptr;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef tp_getset[] =
{
  {
    (char *) "files",
    (getter) tp_getter_files,
    (setter) 0,
    (char *) "registry files",
    NULL
  },
  {
    (char *) "keys",
    (getter) tp_getter_keys,
    (setter) 0,
    (char *) "root keys",
    NULL
  },
  {NULL, NULL, NULL, NULL, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: add_file_by_path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_add_file_by_path (forensics_registry_registry_o *self, PyObject *args)
{
  // parse input args
  const char * arg_role;
  const char * arg_path;
  const char * arg_localpath;

  if (!PyArg_ParseTuple (args, "sss", &arg_role, &arg_path, &arg_localpath))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Forensics_Registry_Registry_File_from_cpp (self->obj->add_file_by_path (arg_role, arg_path, arg_localpath));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: add_file_by_url
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_add_file_by_url (forensics_registry_registry_o *self, PyObject *args)
{
  // parse input args
  const char * arg_role;
  const char * arg_path;
  const char * arg_url;

  if (!PyArg_ParseTuple (args, "sss", &arg_role, &arg_path, &arg_url))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Forensics_Registry_Registry_File_from_cpp (self->obj->add_file_by_url (arg_role, arg_path, arg_url));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: remove_file
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_remove_file (forensics_registry_registry_o *self, PyObject *args)
{
  // parse input args
  std::uint32_t arg_uid;

  if (!PyArg_ParseTuple (args, "I", &arg_uid))
    return nullptr;

  // execute C++ function
  try
    {
      self->obj->remove_file (arg_uid);
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: get_key_by_path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_key_by_path (forensics_registry_registry_o *self, PyObject *args)
{
  // parse input args
  const char * arg_path;

  if (!PyArg_ParseTuple (args, "s", &arg_path))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Forensics_Registry_Registry_Key_from_cpp (self->obj->get_key_by_path (arg_path));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: get_key_by_mask
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_key_by_mask (forensics_registry_registry_o *self, PyObject *args)
{
  // parse input args
  const char * arg_mask;

  if (!PyArg_ParseTuple (args, "s", &arg_mask))
    return nullptr;

  // execute C++ function
  PyObject *ret = PyList_New (0);
  if (!ret)
    return nullptr;

  try
    {
      for (auto k : self->obj->get_key_by_mask (arg_mask))
        {
          PyObject *item = PyMobius_Forensics_Registry_Registry_Key_from_cpp (k);

          if (!item)
            {
              Py_CLEAR (ret);
              return nullptr;
            }

          PyList_Append (ret, item);
          Py_DECREF (item);
        }
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      Py_CLEAR (ret);
      ret = nullptr;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: get_value_by_path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_value_by_path (forensics_registry_registry_o *self, PyObject *args)
{
  // parse input args
  const char * arg_path;

  if (!PyArg_ParseTuple (args, "s", &arg_path))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Forensics_Registry_Registry_Value_from_cpp (self->obj->get_value_by_path (arg_path));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: get_value_by_mask
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_value_by_mask (forensics_registry_registry_o *self, PyObject *args)
{
  // parse input args
  const char * arg_mask;

  if (!PyArg_ParseTuple (args, "s", &arg_mask))
    return nullptr;

  // execute C++ function
  PyObject *ret = PyList_New (0);
  if (!ret)
    return nullptr;

  try
    {
      for (auto v : self->obj->get_value_by_mask (arg_mask))
        {
          PyObject *item = PyMobius_Forensics_Registry_Registry_Value_from_cpp (v);

          if (!item)
            {
              Py_CLEAR (ret);
              return nullptr;
            }

          PyList_Append (ret, item);
          Py_DECREF (item);
        }
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      Py_CLEAR (ret);
      ret = nullptr;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: get_data_by_path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_data_by_path (forensics_registry_registry_o *self, PyObject *args)
{
  // parse input args
  const char * arg_path;

  if (!PyArg_ParseTuple (args, "s", &arg_path))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Forensics_Registry_Registry_Data_from_cpp (self->obj->get_data_by_path (arg_path));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: get_data_by_mask
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_data_by_mask (forensics_registry_registry_o *self, PyObject *args)
{
  // parse input args
  const char * arg_mask;

  if (!PyArg_ParseTuple (args, "s", &arg_mask))
    return nullptr;

  // execute C++ function
  PyObject *ret = PyList_New (0);
  if (!ret)
    return nullptr;

  try
    {
      for (auto d : self->obj->get_data_by_mask (arg_mask))
        {
          PyObject *item = PyMobius_Forensics_Registry_Registry_Data_from_cpp (d);

          if (!item)
            {
              Py_CLEAR (ret);
              return nullptr;
            }

          PyList_Append (ret, item);
          Py_DECREF (item);
        }
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      Py_CLEAR (ret);
      ret = nullptr;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: get_syskey
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_syskey (forensics_registry_registry_o *self, PyObject *)
{
  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_from_bytearray (self->obj->get_syskey ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry: methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef tp_methods[] =
{
  {
    (char *) "add_file_by_path",
    (PyCFunction) tp_f_add_file_by_path,
    METH_VARARGS,
    "add registry file by local path"
  },
  {
    (char *) "add_file_by_url",
    (PyCFunction) tp_f_add_file_by_url,
    METH_VARARGS,
    "add registry file by URL"
  },
  {
    (char *) "remove_file",
    (PyCFunction) tp_f_remove_file,
    METH_VARARGS,
    "remove registry file from registry"
  },
  {
    (char *) "get_key_by_path",
    (PyCFunction) tp_f_get_key_by_path,
    METH_VARARGS,
    "get key by path"
  },
  {
    (char *) "get_key_by_mask",
    (PyCFunction) tp_f_get_key_by_mask,
    METH_VARARGS,
    "get keys by mask"
  },
  {
    (char *) "get_value_by_path",
    (PyCFunction) tp_f_get_value_by_path,
    METH_VARARGS,
    "get value by path"
  },
  {
    (char *) "get_value_by_mask",
    (PyCFunction) tp_f_get_value_by_mask,
    METH_VARARGS,
    "get values by mask"
  },
  {
    (char *) "get_data_by_path",
    (PyCFunction) tp_f_get_data_by_path,
    METH_VARARGS,
    "get data by path"
  },
  {
    (char *) "get_data_by_mask",
    (PyCFunction) tp_f_get_data_by_mask,
    METH_VARARGS,
    "get data by mask"
  },
  {
    (char *) "get_syskey",
    (PyCFunction) tp_f_get_syskey,
    METH_VARARGS,
    "get syskey"
  },
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief mobius.forensics_registry.registry: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject forensics_registry_registry_t =
{
  PyVarObject_HEAD_INIT (NULL, 0)
  "mobius.forensics.registry.registry",          		// tp_name
  sizeof (forensics_registry_registry_o),        		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) tp_dealloc,                 		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "Windows' registry class",               		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  tp_methods,                              		// tp_methods
  0,                                       		// tp_members
  tp_getset,                               		// tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  tp_new,                                  		// tp_new
  0,                                       		// tp_free
  0,                                       		// tp_is_gc
  0,                                       		// tp_bases
  0,                                       		// tp_mro
  0,                                       		// tp_cache
  0,                                       		// tp_subclasses
  0,                                       		// tp_weaklist
  0,                                       		// tp_del
  0,                                       		// tp_version_tag
};
