// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "crypt.h"
#include "cipher_des.h"
#include "hash_md4.h"
#include "hmac.h"
#include <mobius/charset.h>
#include <mobius/string_functions.h>

namespace mobius
{
namespace crypt
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate LM hash
//! \param password password
//! \return LM hash as hex-string
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
generate_hash_lm (const std::string& password)
{
  const std::string u_password = mobius::string::toupper (password);
  mobius::bytearray lm_hash;

  mobius::bytearray key1 (u_password.substr (0, 7));
  key1.rpad (7);
  mobius::crypt::cipher_des des1 (key1);
  lm_hash = des1.encrypt ("KGS!@#$%");

  mobius::bytearray key2 (u_password.length () > 7 ? u_password.substr (7, 7) : "");
  key2.rpad (7);
  mobius::crypt::cipher_des des2 (key2);
  lm_hash += des2.encrypt ("KGS!@#$%");

  return lm_hash;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate NT hash
//! \param password password
//! \return NT hash
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
generate_hash_nt (const std::string& password)
{
  mobius::crypt::hash_md4 md4;
  md4.update (mobius::conv_charset (password, "UTF-8", "UTF-16LE"));
  return md4.get_digest ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate MSDCC1 hash
//! \param password password
//! \param username user name
//! \return MSDCC1 hash
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
generate_hash_msdcc1 (const std::string& password, const std::string& username)
{
  mobius::crypt::hash_md4 md4;
  md4.update (generate_hash_nt (password));
  md4.update (mobius::conv_charset (mobius::string::tolower (username), "UTF-8", "UTF-16LE"));
  return md4.get_digest ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief generate MSDCC2 hash
//! \param password password
//! \param username user name
//! \param iterations number of iterations
//! \return MSDCC2 hash
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::bytearray
generate_hash_msdcc2 (const std::string& password, const std::string& username, int iterations)
{
  const mobius::bytearray pass = generate_hash_msdcc1 (password, username);
  const mobius::bytearray salt = mobius::conv_charset (mobius::string::tolower (username), "UTF-8", "UTF-16LE");

  mobius::crypt::hmac hmac_sha1 (pass, "sha1");
  hmac_sha1.update (salt);
  hmac_sha1.update ({0, 0, 0, 1});

  mobius::bytearray temp = hmac_sha1.get_digest ();
  mobius::bytearray out = temp.slice (0, 15);

  for (int i = 1; i < iterations; i++)
    {
      mobius::crypt::hmac hmac_sha1 (pass, "sha1");
      hmac_sha1.update (temp);
      temp = hmac_sha1.get_digest ();
      out ^= temp.slice (0, 15);
    }

  return out;
}

} // namespace crypt
} // namespace mobius
