#ifndef MOBIUS_CRYPT_CIPHER_BLOCK_H
#define MOBIUS_CRYPT_CIPHER_BLOCK_H

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/crypt/cipher_base.h>
#include <mobius/crypt/cipher_block_mode.h>
#include <mobius/bytearray.h>
#include <algorithm>
#include <cstdint>
#include <cstddef>

namespace mobius
{
namespace crypt
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Abstract base class for block ciphers
//! \author Eduardo Aguiar
//! \see http://en.wikipedia.org/wiki/Block_cipher
//! \see http://en.wikipedia.org/wiki/Block_cipher_mode_of_operation#Cipher_Block_Chaining_.28CBC.29
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class cipher_block : public cipher_base
{
public:
  cipher_block (std::size_t, const std::string&, const mobius::bytearray&);
  void reset () override;
  mobius::bytearray encrypt (const mobius::bytearray&) override;
  mobius::bytearray decrypt (const mobius::bytearray&) override;

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Get block size
  //! \return Block size in bytes
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  std::size_t
  get_block_size () const noexcept override
  {
    return block_size_;
  }
  
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Encrypt data block
  //! \param first input iterator to the first element
  //! \param last input iterator to the element after the last one
  //! \param result output iterator
  //! \return output iterator updated
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  template <typename InputIterator, typename OutputIterator> OutputIterator
  encrypt (InputIterator first, InputIterator last, OutputIterator result)
  {
    while (first != last)
      {
        data_[idx_++] = *first;
        ++first;

        if (idx_ == block_size_)
          {
            mode_.encrypt (data_);
            result = std::copy (data_.begin (), data_.end (), result);
            idx_ = 0;
          }
      }

    return result;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  //! \brief Decrypt data block
  //! \param first input iterator to the first element
  //! \param last input iterator to the element after the last one
  //! \param result output iterator
  //! \return output iterator updated
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  template <typename InputIterator, typename OutputIterator> OutputIterator
  decrypt (InputIterator first, InputIterator last, OutputIterator result)
  {
    while (first != last)
      {
        data_[idx_++] = *first;
        ++first;

        if (idx_ == block_size_)
          {
            mode_.decrypt (data_);
            result = std::copy (data_.begin (), data_.end (), result);
            idx_ = 0;
          }
      }

    return result;
  }

  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  // Abstract methods
  // =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  virtual void encrypt_block (mobius::bytearray&) noexcept = 0;
  virtual void decrypt_block (mobius::bytearray&) noexcept = 0;

private:
  mobius::bytearray data_;
  std::size_t block_size_;
  std::size_t idx_;
  cipher_block_mode mode_;
};

} // namespace crypt
} // namespace mobius

#endif
