// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "resource_impl_local.h"
#include <mobius/datetime/datetime.h>
#include <mobius/uri.h>
#include <mobius/exception.inc>
#include <mobius/exception_posix.inc>
#include <mobius/system/user.h>
#include <mobius/system/group.h>
#include <stdexcept>
#include <unistd.h>
#include <sys/stat.h>

namespace mobius
{
namespace io
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief initialize object
//! \param url URL
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
resource_impl_local::resource_impl_local (const std::string& url)
  : url_ (url)
{
  mobius::uri u (url);
  path_ = u.get_path ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if URL resource exists
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
resource_impl_local::exists () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  return exists_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if resource is a regular file
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
resource_impl_local::is_file () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return is_file_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if resource is a folder
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
resource_impl_local::is_folder () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return is_folder_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get resource size
//! \return size in bytes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
resource_impl_local::size_type
resource_impl_local::get_size () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return size_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get resource owner's user id
//! \return user ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
resource_impl_local::user_id_type
resource_impl_local::get_user_id () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return user_id_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get resource owner's user name
//! \return user name
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
resource_impl_local::get_user_name () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return user_name_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get resource group ID
//! \return group ID
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
resource_impl_local::group_id_type
resource_impl_local::get_group_id () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return group_id_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get resource owner's group name
//! \return group name
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::string
resource_impl_local::get_group_name () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return group_name_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get file access permissions
//! \return permission mask
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
resource_impl_local::permission_type
resource_impl_local::get_permissions () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return permissions_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get last access timestamp
//! \return last resource access time
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
resource_impl_local::get_last_access_time () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return last_access_time_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get last metadata modification timestamp
//! \return last resource metadata modification time
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
resource_impl_local::get_last_modification_time () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return last_modification_time_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get last metadata modification timestamp
//! \return last resource metadata modification time
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::datetime::datetime
resource_impl_local::get_last_metadata_time () const
{
  if (!is_stat_loaded_)
    _load_stat ();

  if (!exists_)
    throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("resource does not exist"));

  return last_metadata_time_;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief load metadata calling POSIX stat function
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
resource_impl_local::_load_stat () const
{
  if (is_stat_loaded_)
    return;

  is_stat_loaded_ = true;
  struct stat st;

  if (stat (path_.c_str (), &st))   // error
    {
      exists_ = false;

      if (errno != ENOENT && errno != ENOTDIR)
        throw std::runtime_error (MOBIUS_EXCEPTION_POSIX);
    }

  else
    {
      exists_ = true;
      is_file_ = S_ISREG (st.st_mode);
      is_folder_ = S_ISDIR (st.st_mode);
      size_ = st.st_size;
      user_id_ = st.st_uid;
      group_id_ = st.st_gid;
      permissions_ = st.st_mode & 0777;
      last_access_time_ = mobius::datetime::new_datetime_from_unix_timestamp (st.st_atime);
      last_modification_time_ = mobius::datetime::new_datetime_from_unix_timestamp (st.st_mtime);
      last_metadata_time_ = mobius::datetime::new_datetime_from_unix_timestamp (st.st_ctime);

      mobius::system::user user (user_id_);
      if (user)
        user_name_ = user.get_name ();

      mobius::system::group group (group_id_);
      if (group)
        group_name_ = group.get_name ();
    }
}

} // namespace io
} // namespace mobius
