// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief C++ API module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "registry_key.h"
#include "registry_value.h"
#include "registry_data.h"
#include "api.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief registry_key: tp_new (default constructor)
//! \param type type object
//! \param args argument list
//! \param kwds keywords dict
//! \return new registry_key object or None if object is null
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
PyMobius_Ant_Registry_Registry_Key_from_cpp (mobius::ant::registry::registry_key r)
{
  PyObject *ret = nullptr;
  
  if (r)
    {
      ret = _PyObject_New (&ant_registry_registry_key_t);
  
      if (ret)
        ((ant_registry_registry_key_o *) ret)->obj = new mobius::ant::registry::registry_key (r);
    }
  else
    {
      Py_INCREF (Py_None);
      ret = Py_None;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
tp_dealloc (ant_registry_registry_key_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: name getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_name (ant_registry_registry_key_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_FromString (self->obj->get_name ().c_str ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: name setter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_name (ant_registry_registry_key_o *self, PyObject *value, void *)
{
  // if it is a delete operation, returns with error
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'name' attribute");
      return -1;
    }

  // check argument type
  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'name' attribute");
      return -1;
    }

  // set name attribute
  try
    {
      self->obj->set_name (PyString_AsString (value));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }
    
  // return ok
  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: classname getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_classname (ant_registry_registry_key_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_FromString (self->obj->get_classname ().c_str ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: last_modification_time getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_last_modification_time (ant_registry_registry_key_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyDateTime_from_datetime (self->obj->get_last_modification_time ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: parent getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_parent (ant_registry_registry_key_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Ant_Registry_Registry_Key_from_cpp (self->obj->get_parent ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: subkeys getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_subkeys (ant_registry_registry_key_o *self)
{
  PyObject *ret = PyList_New (0);
  if (!ret)
    return nullptr;

  try
    {
      for (auto s : *(self->obj))
        {
          PyObject *item = PyMobius_Ant_Registry_Registry_Key_from_cpp (s);

          if (!item)
            {
              Py_CLEAR (ret);
              return nullptr;
            }

          PyList_Append (ret, item);
        }
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      Py_CLEAR (ret);
      ret = nullptr;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: values getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_values (ant_registry_registry_key_o *self)
{
  PyObject *ret = PyList_New (0);
  if (!ret)
    return nullptr;

  try
    {
      for (auto v : self->obj->get_values ())
        {
          PyObject *item = PyMobius_Ant_Registry_Registry_Value_from_cpp (v);

          if (!item)
            {
              Py_CLEAR (ret);
              return nullptr;
            }

          PyList_Append (ret, item);
        }
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      Py_CLEAR (ret);
      ret = nullptr;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef tp_getset[] =
{
  {
    (char *) "name",
    (getter) tp_getter_name,
    (setter) tp_setter_name,
    (char *) "key name",
    NULL
  },
  {
    (char *) "classname",
    (getter) tp_getter_classname,
    (setter) 0,
    (char *) "class name",
    NULL
  },
  {
    (char *) "last_modification_time",
    (getter) tp_getter_last_modification_time,
    (setter) 0,
    (char *) "last modification date/time",
    NULL
  },
  {
    (char *) "parent",
    (getter) tp_getter_parent,
    (setter) 0,
    (char *) "parent key",
    NULL
  },
  {
    (char *) "subkeys",
    (getter) tp_getter_subkeys,
    (setter) 0,
    (char *) "subkeys",
    NULL
  },
  {
    (char *) "values",
    (getter) tp_getter_values,
    (setter) 0,
    (char *) "values",
    NULL
  },
  {NULL, NULL, NULL, NULL, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: add_key
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_add_key (ant_registry_registry_key_o *self, PyObject *args)
{
  // parse input args
  ant_registry_registry_key_o * arg_key;

  if (!PyArg_ParseTuple (args, "O", &arg_key))
    return nullptr;

  // execute C++ function
  try
    {
      self->obj->add_key (*arg_key->obj);
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: remove_key
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_remove_key (ant_registry_registry_key_o *self, PyObject *args)
{
  // parse input args
  const char * arg_name;

  if (!PyArg_ParseTuple (args, "s", &arg_name))
    return nullptr;

  // execute C++ function
  try
    {
      self->obj->remove_key (arg_name);
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: clear_keys
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_clear_keys (ant_registry_registry_key_o *self, PyObject *)
{
  // execute C++ function
  try
    {
      self->obj->clear_keys ();
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: get_key_by_path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_key_by_path (ant_registry_registry_key_o *self, PyObject *args)
{
  // parse input args
  const char * arg_path;

  if (!PyArg_ParseTuple (args, "s", &arg_path))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Ant_Registry_Registry_Key_from_cpp (self->obj->get_key_by_path (arg_path));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: get_key_by_name
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_key_by_name (ant_registry_registry_key_o *self, PyObject *args)
{
  // parse input args
  const char * arg_path;

  if (!PyArg_ParseTuple (args, "s", &arg_path))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Ant_Registry_Registry_Key_from_cpp (self->obj->get_key_by_name (arg_path));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: get_value_by_path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_value_by_path (ant_registry_registry_key_o *self, PyObject *args)
{
  // parse input args
  const char * arg_path;

  if (!PyArg_ParseTuple (args, "s", &arg_path))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Ant_Registry_Registry_Value_from_cpp (self->obj->get_value_by_path (arg_path));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: get_value_by_name
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_value_by_name (ant_registry_registry_key_o *self, PyObject *args)
{
  // parse input args
  const char * arg_path;

  if (!PyArg_ParseTuple (args, "s", &arg_path))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Ant_Registry_Registry_Value_from_cpp (self->obj->get_value_by_name (arg_path));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: get_data_by_path
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_data_by_path (ant_registry_registry_key_o *self, PyObject *args)
{
  // parse input args
  const char * arg_path;

  if (!PyArg_ParseTuple (args, "s", &arg_path))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Ant_Registry_Registry_Data_from_cpp (self->obj->get_data_by_path (arg_path));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: get_data_by_name
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_data_by_name (ant_registry_registry_key_o *self, PyObject *args)
{
  // parse input args
  const char * arg_path;

  if (!PyArg_ParseTuple (args, "s", &arg_path))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyMobius_Ant_Registry_Registry_Data_from_cpp (self->obj->get_data_by_name (arg_path));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: has_subkeys
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_has_subkeys (ant_registry_registry_key_o *self, PyObject *)
{
  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyBool_FromLong (self->obj->has_subkeys ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief registry_key: methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef tp_methods[] =
{
  {
    (char *) "add_key",
    (PyCFunction) tp_f_add_key,
    METH_VARARGS,
    "add a subkey"
  },
  {
    (char *) "remove_key",
    (PyCFunction) tp_f_remove_key,
    METH_VARARGS,
    "remove subkey"
  },
  {
    (char *) "clear_keys",
    (PyCFunction) tp_f_clear_keys,
    METH_VARARGS,
    "clear all keys"
  },
  {
    (char *) "get_key_by_path",
    (PyCFunction) tp_f_get_key_by_path,
    METH_VARARGS,
    "get key by path"
  },
  {
    (char *) "get_key_by_name",
    (PyCFunction) tp_f_get_key_by_name,
    METH_VARARGS,
    "get key by name"
  },
  {
    (char *) "get_value_by_path",
    (PyCFunction) tp_f_get_value_by_path,
    METH_VARARGS,
    "get value by path"
  },
  {
    (char *) "get_value_by_name",
    (PyCFunction) tp_f_get_value_by_name,
    METH_VARARGS,
    "get value by name"
  },
  {
    (char *) "get_data_by_path",
    (PyCFunction) tp_f_get_data_by_path,
    METH_VARARGS,
    "get data by path"
  },
  {
    (char *) "get_data_by_name",
    (PyCFunction) tp_f_get_data_by_name,
    METH_VARARGS,
    "get data by name"
  },
  {
    (char *) "has_subkeys",
    (PyCFunction) tp_f_has_subkeys,
    METH_VARARGS,
    "check if key has subkeys"
  },
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief mobius.ant_registry.registry_key: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject ant_registry_registry_key_t =
{
  PyVarObject_HEAD_INIT (NULL, 0)
  "mobius.ant.registry.registry_key",      		// tp_name
  sizeof (ant_registry_registry_key_o),    		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) tp_dealloc,                 		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "registry_key class",                    		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  tp_methods,                              		// tp_methods
  0,                                       		// tp_members
  tp_getset,                               		// tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  0,                                  	         	// tp_new
  0,                                       		// tp_free
  0,                                       		// tp_is_gc
  0,                                       		// tp_bases
  0,                                       		// tp_mro
  0,                                       		// tp_cache
  0,                                       		// tp_subclasses
  0,                                       		// tp_weaklist
  0,                                       		// tp_del
  0,                                       		// tp_version_tag
};
