# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
XML_ENCODING='utf-8'

import libxml2
import os.path

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Category
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Category (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.id = None
    self.icon_data = None
    self.is_dataholder = False
    self.attributes = []

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Attribute
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Attribute (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    self.id = None
    self.name = None
    self.description = None
    self.datatype = 'string'
    self.initvalue = None

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @deprecated Persistence layer for category database prior to 0.4.5
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Pickle (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __get_prop (self, node, name):
    value = node.prop (name)
    if value:
      value = value.decode (XML_ENCODING)
    return value

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set node property with correct encoding
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __set_prop (self, node, name, value):
    if value != None:
      node.setProp (name, value.encode (XML_ENCODING))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load categories
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load (self, path):
    categories = []

    if os.path.exists (path):
      doc = libxml2.parseFile (path)
      node = doc.getRootElement ()
      categories = self.load_categories (node)

    return categories

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <categories>
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_categories (self, node):
    categories = []

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'category':
        cat = self.load_category (node)
        categories.append (cat)

      node = node.next

    return categories

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <category> from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_category (self, node):
    cat = Category ()
    cat.id = self.__get_prop (node, 'id')
    cat.name = self.__get_prop (node, 'name')
    cat.is_dataholder = self.__get_prop (node, 'is_dataholder') == 'True'

    # load children
    node = node.children

    while node:
      if node.type == 'element' and node.name == 'attribute':
        cat.attributes.append (self.load_attribute (node))

      elif node.type == 'element' and node.name == 'icon':
        cat.icon_data = self.load_icon (node)

      node = node.next

    return cat

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <attribute> from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_attribute (self, node):
    attr = Attribute ()
    attr.id = self.__get_prop (node, 'id')
    attr.name = self.__get_prop (node, 'name')
    attr.description = self.__get_prop (node, 'description')
    attr.initvalue = self.__get_prop (node, 'default')

    return attr

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Load <icon> from node
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def load_icon (self, node):
    icon = None
    node = node.children

    while node:
      if node.type == 'text':
        icon = node.getContent ()

      node = node.next

    return icon

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief icon data
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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'''

ICON_NEW_CATEGORY = '''
iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAABGdBTUEAALGPC/xhBQAAAAZiS0dE
AP8A/wD/oL2nkwAAAAlwSFlzAAAN1wAADdcBQiibeAAAAAd0SU1FB9gHEAw6I15U6QIAAAMiSURB
VFjD7ddLaFxlFAfw351HnDxN05gapYoiVjREqSk+0LYQxHapG+tacCHiLmA36kJRXFmtohDEldLW
YAXBVSKE2IoPtKJQxFrFkqTWkpo6ySSdmevi3pDLdB6ZFIqLHPi4353zcc7/O//zuMOGrEsyD3P9
Yfp+of87rnsN/VfJedvTDMxxKORkyImQ0TKbz+O+Zq2lmgfQfj/DndyJa9CLRwIe7cJjVyME3XQf
ZGeeUyFnQrbl6ZjASNNk1tF14vbqqotTzD9BuY0yLpT4dxyTuLficIgTKFWzFNT2n3uJ3H4G5iOm
kqs9x3MdbI1p+Cnkg4sEy5E+afajXjyI401GIIUnW9jVGzmptsL47PaAB7rIxb8nUyvI8+HeWgAa
JGFYsWqdqWsjhbZ1VkHaakhrsRU00F9RGaZjliqBXJnTegCG4oSpAiCdAJKqAyjABL5ttgwzw2w9
EpXL6SGybVFSZWuACGqAebvIC9O0dnGym4UQl9YAoOcABzYxHfL89+RK7IgBVItEusL5yv7VJebe
xZ/ccZDiIn5rBCBNto8t2BxweFPkeCXJ0lVApKpQchrheXzJ3CRzh/BQrKgLIEdruPraKepwYcL4
CoBMYp+qALKEYCm6PXHov1hLEuaZL0ZGL+CZBd4q0BJTkE3kQi0q0tiGcm/kODvEjT9y8yuN6iyW
jltYvIdPljn+MTPYcUM05jMVK1vnORXyR4m+EaYG+WyAmU/xdwMA+XF+v4mZSyweJXuWnbu5tSLs
mYpoJJ3D3ix/DbGvh+Es4yV+nsWxRrNgmXNPoT1+H6EooiGoyIOxEqMFzpUJUgzi5fZoEF6L91oT
d0xHObb2cZxf3ZbjmyUBvFninVPMH8E3OMv0XXz1Okd72NXU0Ms07lPZRK2nUShhAe8n6vsMweNM
76nR3luvYBasVMLK88UW3rib/rHVc1uOMbqHfalm/WTqX//zGHyyz6cwU6LUjdvwK0GJQsB4FTM/
ZNf5SVYcY7KLicHLdYV/KH4dU4HZ3Ty7n/T2y0dkoROz6gzzRpJLdLXSGoZbpsZXy9LG/6kN+V/K
f7T3xDg+BD+/AAAAAElFTkSuQmCC'''

ICON_UNKNOWN_CATEGORY = ''' PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiIHN0YW5kYWxvbmU9Im5vIj8+Cjwh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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'category-model'
    self.name = 'Category Model'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Category model'
    self.mediator = mediator.copy ()
    self.icon_data = ICON_DATA

    # specific data
    self.__loaded = False
    self.__path = self.mediator.call ('app.get-path', 'category.xml')
    self.__categories = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.advertise ('category.new', self.svc_category_new)
    self.mediator.advertise ('category.set', self.svc_category_set)
    self.mediator.advertise ('category.get', self.svc_category_get)
    self.mediator.advertise ('category.remove', self.svc_category_remove)
    self.mediator.advertise ('category.get-icon-data', self.svc_category_get_icon_data)
    self.mediator.advertise ('category.attribute.new', self.svc_category_attribute_new)
    self.mediator.advertise ('category.attribute.get', self.svc_category_attribute_get)
    self.mediator.advertise ('category.attribute.set-name', self.svc_category_attribute_set_name)
    self.mediator.advertise ('category.attribute.set-init-value', self.svc_category_attribute_set_init_value)
    self.mediator.advertise ('category.attribute.remove', self.svc_category_attribute_remove)
    self.mediator.advertise ('category.attribute.index', self.svc_category_attribute_index)
    self.mediator.advertise ('category.attribute.move', self.svc_category_attribute_move)
    self.mediator.advertise ('category.iter-attributes', self.svc_category_iter_attributes)
    self.mediator.advertise ('category.list.iter', self.svc_category_list_iter)
    self.mediator.advertise ('category.list.save', self.svc_category_list_save)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Load on demand
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def load_on_demand (self):
    if not self.__loaded:

      # @deprecated handle category.xml prior to 0.4.5
      data = open (self.__path).read ()

      if '<categories>' in data:
        pickle = Pickle ()
        self.__categories = {}
        self.__categories.update (((cat.id, cat) for cat in pickle.load (self.__path)))

      else:
        self.__categories = self.mediator.call ('xml.unpickle', self.__path)

      self.__loaded = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.new
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_new (self, cat_id):
    self.load_on_demand ()

    category = Category ()
    category.id = cat_id
    category.icon_data = ICON_NEW_CATEGORY
    self.svc_category_set (category)

    return category

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.set
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_set (self, category):
    self.load_on_demand ()
    self.__categories[category.id] = category

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.get
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_get (self, cat_id):
    self.load_on_demand ()
    return self.__categories.get (cat_id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  # @brief Service: category.remove
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_remove (self, cat_id):
    self.load_on_demand ()
    self.__categories.pop (cat_id, None)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.get-icon-data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_get_icon_data (self, cat_id):
    cat = self.svc_category_get (cat_id)

    if cat:
      icon_data = cat.icon_data
    else:
      icon_data = ICON_UNKNOWN_CATEGORY

    return icon_data

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  # @brief Service: category.attribute.new
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_attribute_new (self, cat_id, attr_id):
    cat = self.svc_category_get (cat_id)
    if cat:
      a = Attribute ()
      a.id = attr_id
      cat.attributes.append (a)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  # @brief Service: category.attribute.get
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_attribute_get (self, cat_id, attr_id):
    cat = self.svc_category_get (cat_id)
    if cat:
      for a in cat.attributes:
        if a.id == attr_id:
          return a

    return None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  # @brief Service: category.attribute.set-name
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_attribute_set_name (self, cat_id, attr_id, name):
    attr = self.svc_category_attribute_get (cat_id, attr_id)
    if attr:
      attr.name = name

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  # @brief Service: category.attribute.set-init-value
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_attribute_set_init_value (self, cat_id, attr_id, initvalue):
    attr = self.svc_category_attribute_get (cat_id, attr_id)
    if attr:
      attr.initvalue = initvalue

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  # @brief Service: category.attribute.remove
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_attribute_remove (self, cat_id, attr_id):
    cat = self.svc_category_get (cat_id)
    if cat:
      cat.attributes = [a for a in cat.attributes if a.id != attr_id]

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  # @brief Service: category.attribute.index
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_attribute_index (self, cat_id, attr_id):
    cat = self.svc_category_get (cat_id)
    pos = -1

    if cat:
      for idx, a in enumerate (cat.attributes):
        if a.id == attr_id:
          pos = idx

    return pos

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  # @brief Service: category.attribute.move
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_attribute_move (self, cat_id, attr_id, pos):
    cat = self.svc_category_get (cat_id)
    if cat:
      curpos = self.svc_category_attribute_index (cat_id, attr_id)
      if curpos != -1 and curpos != pos:
        attr = cat.attributes[curpos]
        cat.attributes = cat.attributes[:curpos] + cat.attributes[curpos+1:]
        cat.attributes = cat.attributes[:pos] + [attr] + cat.attributes[pos:]

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
  # @brief Service: category.iter-attributes
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_iter_attributes (self, cat_id):
    cat = self.svc_category_get (cat_id)
    if cat:
      return iter (cat.attributes)
    else:
      return iter ([])

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.list.iter
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_list_iter (self):
    self.load_on_demand ()
    return self.__categories.itervalues ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Service: category.list.save
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_category_list_save (self):
    self.load_on_demand ()
    self.mediator.call ('xml.pickle', self.__path, self.__categories)
