/* Gpremacy : A Free implementation of the 1984 board game 
 * Supremacy by Supremacy Games, Inc.
 * Copyright (C) 2005 James C. Jones
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

// created on 08/28/2005 at 14:54
using Gtk;
using System;
using System.Collections;
using Gpremacy.Network;

namespace Gpremacy {
class Game {
	ArrayList players; // of Player
	ArrayList localplayers; // of Player, those who are on the local system.	
	ArrayList cards; // of ResourceCard
	ArrayList allcards; // of ResourceCard
	ArrayList allunits; // of Unit
	ArrayList allresources; // of Resource
	
    Hashtable mapTerritoryHash; // of MapTerritory
    Hashtable territoryHash; // of Territory
    Hashtable unitHash; // of Unit
    Hashtable widgetData; // Really freaking huge. This is bad.
	
	IEnumerator cardsIterator;
	Player playerNobody;
	Territory placeNowhere;
	GameState state;
	GpremacyMarket market;
	GpremacyGUI mainGUI;
	bool justShuffledResourceCards;
	string mapFileName;
	
	public GameLink gameLink; // for net play
	
	private static Game instance;
	private static int numOfReference;
	private static int unitCounter;	
	
	private Game()
	{
		numOfReference = 0;
		unitCounter = 0;		
	}
	
	public string VersionString
	{
		get { return "20051117: Multiplay Release 2"; }
	}
	
	public void init()
	{
		cards = new ArrayList();
		allcards = new ArrayList();
		allunits = new ArrayList();
		allresources = new ArrayList();

		playerNobody = new Player(-1, "Nobody", new Gdk.Color(123,123,123));
		placeNowhere = new Territory(-1, "Nowhere", playerNobody.CountryID, playerNobody);

		allunits.Add(new Army(playerNobody, placeNowhere));
		allunits.Add(new Navy(playerNobody, placeNowhere));
		allunits.Add(new Nuke(playerNobody)); 
		allunits.Add(new LSat(playerNobody));
		
		allresources.Add(new Oil());
		allresources.Add(new Minerals());
		allresources.Add(new Grain());

		players = new ArrayList();		
		setupPlayers();
		localplayers = new ArrayList();
		
		territoryHash = new Hashtable();
		mapTerritoryHash = new Hashtable();
		unitHash = new Hashtable();
		widgetData = new Hashtable();
						
		market = new GpremacyMarket();
				
		foreach (Resource r in allresources)
		{
			market.initResource(r, 12);
		}
			
		state = new GameState(this);		

		// Before starting GUI, load CSV files
		ArrayList territories = LoadCountryBoundaries();
		LoadResourceCards(territories);

		mainGUI = new GpremacyGUI(this);
		
		try {
			mainGUI.init(territories);
		} catch (Exception e) {
			Console.WriteLine(e + " caught!");
			
			new ExceptionWindow("Gpremacy", e, GUI.mainWindow);
		}		
	}
	
	public static Game GetInstance()
	{
		if(instance == null)
		{
			instance = new Game();
		}
		numOfReference++;
		return instance;
	}           

	public static int Reference
	{
		get { return numOfReference; }
	} 
	
	
	public static void Main(string[] args)
	{
		Game gameObject = Game.GetInstance();	
		gameObject.init();			
	}

	/* Static Helpers */
	
	public static int DefaultPort
	{
		get { return 34543; }
	}
	
	public static int nextUnitCount()
	{
		System.Console.WriteLine("nextUnitCount: Handing out unit id of " + unitCounter);
		return unitCounter++;
	}
	
	public static object getObjectFromDataIn(Gtk.Widget wid)
	{
		Hashtable table = Game.GetInstance().WidgetData;

		foreach(DictionaryEntry de in table)
		{
			if ( (de.Key is Gtk.Widget) && ( (Gtk.Widget)de.Key == wid ) )
			{
				return de.Value;
			}
		}
		return null;
	}
	
	public static void putObjectDataIn(Gtk.Widget wid, object obj)
	{
		Game.GetInstance().WidgetData.Add(wid, obj);
	}	
	
	/* End Static Helpers */
	/* Helping Hashtables */
	
	public Hashtable MapTerritoryHashTable
	{
		get { return mapTerritoryHash; }
	}
	
	public Hashtable TerritoryHashTable
	{
		get { return territoryHash; }
	}
	
	public Hashtable UnitHashTable
	{
		get { return unitHash; }
	}
	
	public Hashtable WidgetData
	{
		get { return widgetData; } 
	}
	
	public Unit UnitByID(int id)
	{		
       	if (!unitHash.ContainsKey(id)) {
       		throw new ArgumentOutOfRangeException("Asked for a Unit with an ID of " + id +  " which is out of range.");
       	}
      	return (Unit)unitHash.get_Item(id);		
	}
	
	public Unit GetLocalCopyOfUnit(Unit u)
	{				
		return UnitByID(u.ID);
	}

    public MapTerritory MapTerritoryByID(int id)
    {
       	if (!mapTerritoryHash.ContainsKey(id)) {
       		throw new ArgumentOutOfRangeException("Asked for a MapTerritory with an ID of " + id +  " which is out of range.");
       	}
      	return (MapTerritory)mapTerritoryHash.get_Item(id);
    }
    
    public Territory TerritoryByName(string name)
    {
       	if (!territoryHash.ContainsKey(name)) {
       		throw new ArgumentOutOfRangeException("Could not find the territory by name of " + name +". Check your countries.csv file; maybe you mispelled it?");
       	}
      	return (Territory)territoryHash.get_Item(name);
    }    
	
	public Player PlayerByName(string name)
	{
		foreach(Player p in players)
			if (p.Name == name)
				return p;
		return null;
	}
	
	public Player GetLocalCopyOfPlayer(Player p)
	{
		foreach (Player a in players)
			if (p == a)
				return a;
		return null;
	}		
	
	/* End Hashtables */

	public Player PlayerNobody
	{
		get { return playerNobody; }
	}
	public Territory PlaceNowhere
	{
		get { return placeNowhere; }
	}	

	public ArrayList Players
	{
		get { return players; }
	//	set { players = value; }
	}
	
	public ArrayList LocalPlayers
	{
		get { return localplayers; }
	}
	
	public GpremacyGUI GUI
	{
		get { return mainGUI; }
	}
	
	public GpremacyMarket Market
	{
		get { return market; }
	}
	
	public GameState State
	{
		get { return state; }
	}	
			
	public ArrayList AllCards
	{
		get { return allcards; }
	//	set { cards = value; }
	}
	
	public ArrayList Cards
	{
		get { return cards; }
	//	set { cards = value; }
	}
	
	public ArrayList AllUnits
	{
		get { return allunits; }
	}
	
	public ArrayList AllResources
	{
		get { return allresources; }
	}
	
	public string MapFileName
	{
		get { return mapFileName; }
	}
		
	void setupPlayers()
	{
		players.Add(new Player(1, "United States of America", new Gdk.Color(8,75,8)));
		players.Add(new Player(2, "Confederacy of South America", new Gdk.Color(22,158,22)));
		players.Add(new Player(3, "Federation of African States", new Gdk.Color(157,31,200)));
		players.Add(new Player(4, "League of European Nations", new Gdk.Color(179,94,11)));
		players.Add(new Player(5, "Union of Soviet Sovereign Republics", new Gdk.Color(178,11,93)));
		players.Add(new Player(6, "People's Republic of China", new Gdk.Color(178,137,11)));
	}
	
	public void Quit()	
	{
		if (gameLink != null)
			gameLink.stop();
		Application.Quit();
	}
	
	public void HaltGame(String a)
	{
		if (gameLink != null)
			gameLink.stop();
		throw new Exception("Game halting because of " + a);
	}	
	
	public bool hasSufficientWeath(Player p, Dictionary dict, int cash, bool ShowErrors)
	{
		if (dict != null)
		foreach(DictionaryEntry d in dict.Data)
		{
			/* If charging multiple stocks, this will possibly subtract part of a cost set */
			if (p.getStockpileAmount((Resource)d.Key) < -1*(Int32)d.Value)
			{
				if (ShowErrors)
					Game.GetInstance().GUI.ShowWarning("You do not have enough " + ((Resource)d.Key).Name );
				return false;
			}
		}
		if ( p.Money < cash )
		{
			if (ShowErrors)
				Game.GetInstance().GUI.ShowWarning("You do not have enough money");
			return false;		
		}
		return true;	
	}
	
	public bool hasSufficientWeath(Player p, ArrayList stocks, int cash, bool ShowErrors)
	{
		if (stocks != null)
		foreach(Stock s in stocks)
		{
			/* If charging multiple stocks, this will possibly subtract part of a cost set */
			if (p.getStockpileAmount(s.Good) < -1*s.Number)
			{
				if (ShowErrors)
					Game.GetInstance().GUI.ShowWarning("You do not have enough " + s.Good.Name);
				return false;
			}
		}
		if ( p.Money < cash )
		{
			if (ShowErrors)
				Game.GetInstance().GUI.ShowWarning("You do not have enough money");
			return false;		
		}
		return true; 					
	}

	public void GiveInitialUnits() 
	{
		Army unit;
		foreach(Territory territory in GUI.Map.Territories)
		{
			if (territory.IsLand && territory.Owner != PlayerNobody && territory.Owner.Active)
			{				
				unit = new Army(territory.Owner, territory);				
				Orig_BuildUnit cmd = new Orig_BuildUnit(unit, territory, territory.Owner);
				
				cmd.addToID(9000); // Note that it was given by the server at start.
				state.Execute(cmd);
			}
		}
	}

	public bool JustShuffledResourceCards
	{
		get { return justShuffledResourceCards; }
		set { justShuffledResourceCards = value; }
	}

	public ResourceCard CurrentResourceCard
	{
		get {
			return (ResourceCard)cardsIterator.Current;
		}
	}
	
	public void NextResourceCard()
	{
		if (cardsIterator.MoveNext() == false)
		{
			cardsIterator.Reset();
			cardsIterator.MoveNext();
		}
	}
	
	public void RemoveResourceCard(ResourceCard c) 
	{
		cards.Remove(c);
		cardsIterator = cards.GetEnumerator();
		cardsIterator.Reset();
		cardsIterator.MoveNext();
	}
	
	public void PopCurrentResourceCard()
	{		
		cards.Remove(cardsIterator.Current);
		cardsIterator = cards.GetEnumerator();
		cardsIterator.Reset();
		cardsIterator.MoveNext();
	}
	
	public void ShuffleResourceCards()	
	{
		System.Console.WriteLine("Count of AllCards: " + allcards.Count);
		System.Console.WriteLine("Count of UnclaimedCards: " + cards.Count);

		justShuffledResourceCards = true;
						
		Random r = new Random();

		/* Algorithm: 
  		RandomInx : integer;
  		TempPtr   : pointer;

  		for Inx := aList.Count - 1 downto 1 do begin
	    	RandomInx := Random(Inx + 1);
    		if (RandomInx <> Inx) then begin
	      		TempPtr := aList[Inx];
	      		aList[Inx] := aList[RandomInx];
	      		aList[RandomInx] := TempPtr;
	    	end;
		*/
		
		object TempPtr;
		int RandomIndex;
		int Index;
		for (Index = cards.Count - 1; Index > 0; Index--)
		{
			RandomIndex = r.Next() % Index;
			if (RandomIndex != Index)
			{
				TempPtr = cards[Index];
				cards[Index] = cards[RandomIndex];
				cards[RandomIndex] = TempPtr;
			}
		}
		
		cardsIterator = cards.GetEnumerator();
		cardsIterator.Reset();
		cardsIterator.MoveNext();
		
	}
	
	public void DistributeResourceCards()
	{
		foreach(ResourceCard card in allcards)
		{
			foreach(Player player in players)
			{
				if ((card.isResource()) && (card.Place.Owner == player) && (player.Active))
				{
					player.addResourceCard(card);
					card.Active = true; // mark it active 
					cards.Remove(card); // take it from the normal deck
					break;
				}
			}
		}
	}
	
   private ArrayList LoadCountryBoundaries ()
   {
   	String line;
   	String name;
   	Player owner;
   	int ownerid, x, y, id = 0, centerX = 0, centerY = 0;
   	ArrayList points = new ArrayList(); // of Gdk.Point
   	ArrayList graphConnections = new ArrayList(); // of ArrayList of String.
   	ArrayList subGraph; // of String
   	bool isLand;
   	Territory lastTerritory = null;
   	MapTerritory mapTerritory = null;
   	mapFileName= "";
   	
   	ArrayList territories = new ArrayList();
   	
   	try {
        	System.IO.StreamReader input = new System.IO.StreamReader(SupportFileLoader.locateGameFile("countries.csv"));
	       	do {
	       		line = input.ReadLine();
       			if ((line == null) || (line.Length > 0 && line[0]=='#')) continue;
	       		
	       		// First line is the map path
	       		if (mapFileName.Length < 1)
	       		{
	       			mapFileName = line;
	       			continue;
	       		}
	       		
       			// Format: Name,   Owner#, [x, y]*
       			//         string  int      int,int
       			// Format: \t%ConnectingName, ConnectingName*
				//		      string          string
       			if (line.IndexOf(",") <= 0)
       				continue;
       			if (line.IndexOf("%") <= 0) 
				{
       				String[] parts;
	       			parts = line.Split(',');
       				/*for (int i=0;i<parts.Length;i++) 
       					System.Console.Write("[" + parts[i] + "],");
       				System.Console.WriteLine(); */
       				
	       			if (parts.Length > 2) 
       				{
       					centerX = centerY = 0;
       					
	       				// Extract data
       					name = parts[0].Trim('"');
       					ownerid = Int16.Parse(parts[1]);
	        			if (ownerid > 0 && ownerid <= Players.Count)
	        				owner = (Player)(Players[ownerid-1]);
	        			else
	        				owner = PlayerNobody;
	        			int offset = 0;
	        			if (parts[2] != "false" && parts[2] != "true")
	        			{
	        				offset = 2;
	        				// extract centers
	        				centerX = Int16.Parse(parts[2]);
	        				centerY = Int16.Parse(parts[3]); 	        				
	        			}
	        			isLand = Boolean.Parse(parts[offset+2]);
	        			// Get points
	        			for (int i=offset+3; i<parts.Length-1; i=i+2)
	        			{
	        				x = Int16.Parse(parts[i]); 
	        				y = Int16.Parse(parts[i+1]); 
	        				points.Add(new Gdk.Point(x,y));
	        			} 

						lastTerritory = new Territory(id, name, ownerid, owner);
						territories.Add(lastTerritory);
						TerritoryHashTable.Add(name, lastTerritory);

						mapTerritory = new MapTerritory(name, isLand, points);
						if (centerX > 0)
						{
							mapTerritory.centerX = centerX;
							mapTerritory.centerY = centerY;
						}
						MapTerritoryHashTable.Add(id, mapTerritory);
						
						id++;
						
                   	points.Clear();
       				}
       			} else {
       				/* Read in Graph*/	       				
       				String[] parts;
	       			parts = line.Substring(line.IndexOf("%")+1).Split(',');
	       			subGraph = new ArrayList();
	       			subGraph.Add(lastTerritory);
	       			for (int i=0; i<parts.Length; i++)
	       				subGraph.Add(parts[i]);
	       			graphConnections.Add(subGraph);
       			}
       		} while (input.Peek() > -1);
       		input.Close();
       		
  		} catch ( System.IO.FileNotFoundException e ) {
   			HaltGame("Couldn't open countries.csv. Game halts.");    		       			
  		}
  		
  		/* Take care of connection graphs */       		
  		Territory home, targetTerritory;
  		String target;
  		bool noLand;
  		foreach (ArrayList graph in graphConnections) {
  			noLand = true;
  			home = (Territory)graph[0];
  			
  			for (int i=1; i<graph.Count; i++) {
  				target=(String)graph[i];
  				targetTerritory = TerritoryByName(target);
  				if (targetTerritory.IsLand)
  					noLand = false;
  				home.MapTerritory.addConnection(targetTerritory);
  			}
  			
  			if (noLand)
  				home.MapTerritory.deepSea = true;
  		}
  		
  		return territories;
   }	
	
    public void LoadResourceCards (ArrayList Territories)
    {
       	String line;
       	Resource r;
       	Territory t;
       	Unit u;
       	String name, resourceName, terrName;
       	int amt;
       	ResourceCard newcard;
        	
       	try {
       		
	  		System.IO.StreamReader input = new System.IO.StreamReader(SupportFileLoader.locateGameFile("resource_cards.csv"));
	       	do {
	       		line = input.ReadLine();
	       		if ((line == null) || (line.Length > 0 && line[0]=='#')) continue;
	       		
	       		t = null; r = null; u = null;
	       		
	       		// Format: Business Name, Territory, Resource Name, Resource Amount
	       		//           string         string       string         int	       		
	       		if (line.IndexOf(",") > 0) 
				{
	       			String[] parts;
	       			parts = line.Split(',');
	       		//	for (int i=0;i<parts.Length;i++) 
	       		//		System.Console.WriteLine("got [" + parts[i] + "]");
	       			if (parts.Length > 3) 
	       			{
	       				// Extract data
	       				name = parts[0].Trim('"');
	       				amt = Int16.Parse(parts[3]);
	       				resourceName = parts[2].Trim('"');
	       				terrName = parts[1].Trim('"');
	       				
	       				// Find Resource
	       				if (resourceName.CompareTo("Oil") == 0)
							r = new Oil(playerNobody, amt);
						else if (resourceName.CompareTo("Mineral") == 0)
							r = new Minerals(playerNobody, amt);
						else if (resourceName.CompareTo("Grain") == 0)
							r = new Grain(playerNobody, amt);
						else if (resourceName.CompareTo("Nuke") == 0)
							u = new Nuke(playerNobody);
						else if (resourceName.CompareTo("LSat") == 0)
							u = new LSat(playerNobody);
						else {
	       					System.Console.WriteLine("Could not determine resource type from [" + resourceName + "]");
	       					continue;
	       				}
	       				// Find Territory
	       				foreach (Territory terr in Territories)
	       				{
	       					if (terr.Name.CompareTo(terrName) == 0)
	       					{
	       						t = terr; break;
	       					}
	       				}
	       				if (t == null && u == null)
	       				{
	       					System.Console.WriteLine("Could not determine territory from [" + terrName + "]");
	       					continue;
	       				}
	       				
	       				// Create Card, a Research card or a Resource card
	       				if (u == null) {
	       					newcard = new ResourceCard(r, t, name);
	       					t.addResource(r);
	       				} else { 
	       					newcard = new ResourceCard(u, null, name);
	       				}

	       				allcards.Add(newcard);
	       				cards.Add(newcard);
	       				
	       			
	       			}
	       		}
	       		        		
	      		} while (input.Peek() > -1);
	      		input.Close();
	      		
    	} catch ( System.IO.FileNotFoundException e ) {
       		System.Console.WriteLine("Couldn't open resource_cards.csv. Game halts.");
       		HaltGame("Couldn't open resource_cards.csv. Game halts.");   		       			
    	}
	}

}
}
