/*
 * Decompiled with CFR 0.152.
 */
package org.apache.polaris.core.storage.cache;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Expiry;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.annotations.VisibleForTesting;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import org.apache.iceberg.exceptions.UnprocessableEntityException;
import org.apache.polaris.core.PolarisCallContext;
import org.apache.polaris.core.PolarisDiagnostics;
import org.apache.polaris.core.config.FeatureConfiguration;
import org.apache.polaris.core.config.RealmConfig;
import org.apache.polaris.core.entity.PolarisEntity;
import org.apache.polaris.core.entity.PolarisEntityType;
import org.apache.polaris.core.persistence.dao.entity.ScopedCredentialsResult;
import org.apache.polaris.core.storage.AccessConfig;
import org.apache.polaris.core.storage.PolarisCredentialVendor;
import org.apache.polaris.core.storage.cache.StorageCredentialCacheConfig;
import org.apache.polaris.core.storage.cache.StorageCredentialCacheEntry;
import org.apache.polaris.core.storage.cache.StorageCredentialCacheKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StorageCredentialCache {
    private static final Logger LOGGER = LoggerFactory.getLogger(StorageCredentialCache.class);
    private final PolarisDiagnostics diagnostics;
    private final LoadingCache<StorageCredentialCacheKey, StorageCredentialCacheEntry> cache;

    public StorageCredentialCache(PolarisDiagnostics diagnostics, StorageCredentialCacheConfig cacheConfig) {
        this.diagnostics = diagnostics;
        this.cache = Caffeine.newBuilder().maximumSize(cacheConfig.maxEntries()).expireAfter(Expiry.creating((key, entry) -> {
            long expireAfterMillis = Math.max(0L, Math.min((entry.getExpirationTime() - System.currentTimeMillis()) / 2L, entry.getMaxCacheDurationMs()));
            return Duration.ofMillis(expireAfterMillis);
        })).build(key -> null);
    }

    private long maxCacheDurationMs(RealmConfig realmConfig) {
        Integer cacheDurationSeconds = realmConfig.getConfig(FeatureConfiguration.STORAGE_CREDENTIAL_CACHE_DURATION_SECONDS);
        Integer credentialDurationSeconds = realmConfig.getConfig(FeatureConfiguration.STORAGE_CREDENTIAL_DURATION_SECONDS);
        if (cacheDurationSeconds >= credentialDurationSeconds) {
            throw new IllegalArgumentException(String.format("%s should be less than %s", FeatureConfiguration.STORAGE_CREDENTIAL_CACHE_DURATION_SECONDS.key(), FeatureConfiguration.STORAGE_CREDENTIAL_DURATION_SECONDS.key()));
        }
        return (long)cacheDurationSeconds.intValue() * 1000L;
    }

    public AccessConfig getOrGenerateSubScopeCreds(@Nonnull PolarisCredentialVendor credentialVendor, @Nonnull PolarisCallContext callCtx, @Nonnull PolarisEntity polarisEntity, boolean allowListOperation, @Nonnull Set<String> allowedReadLocations, @Nonnull Set<String> allowedWriteLocations, Optional<String> refreshCredentialsEndpoint) {
        if (!this.isTypeSupported(polarisEntity.getType())) {
            this.diagnostics.fail("entity_type_not_suppported_to_scope_creds", "type={}", new Object[]{polarisEntity.getType()});
        }
        StorageCredentialCacheKey key = StorageCredentialCacheKey.of(callCtx.getRealmContext().getRealmIdentifier(), polarisEntity, allowListOperation, allowedReadLocations, allowedWriteLocations, refreshCredentialsEndpoint);
        LOGGER.atDebug().addKeyValue("key", (Object)key).log("subscopedCredsCache");
        Function<StorageCredentialCacheKey, StorageCredentialCacheEntry> loader = k -> {
            LOGGER.atDebug().log("StorageCredentialCache::load");
            ScopedCredentialsResult scopedCredentialsResult = credentialVendor.getSubscopedCredsForEntity(callCtx, k.catalogId(), polarisEntity.getId(), polarisEntity.getType(), k.allowedListAction(), k.allowedReadLocations(), k.allowedWriteLocations(), k.refreshCredentialsEndpoint());
            if (scopedCredentialsResult.isSuccess()) {
                long maxCacheDurationMs = this.maxCacheDurationMs(callCtx.getRealmConfig());
                return new StorageCredentialCacheEntry(scopedCredentialsResult.getAccessConfig(), maxCacheDurationMs);
            }
            LOGGER.atDebug().addKeyValue("errorMessage", (Object)scopedCredentialsResult.getExtraInformation()).log("Failed to get subscoped credentials");
            throw new UnprocessableEntityException("Failed to get subscoped credentials: %s", new Object[]{scopedCredentialsResult.getExtraInformation()});
        };
        return ((StorageCredentialCacheEntry)this.cache.get((Object)key, loader)).toAccessConfig();
    }

    @Nullable
    @VisibleForTesting
    Map<String, String> getIfPresent(StorageCredentialCacheKey key) {
        return this.getAccessConfig(key).map(AccessConfig::credentials).orElse(null);
    }

    @VisibleForTesting
    Optional<AccessConfig> getAccessConfig(StorageCredentialCacheKey key) {
        return Optional.ofNullable((StorageCredentialCacheEntry)this.cache.getIfPresent((Object)key)).map(StorageCredentialCacheEntry::toAccessConfig);
    }

    private boolean isTypeSupported(PolarisEntityType type) {
        return type == PolarisEntityType.CATALOG || type == PolarisEntityType.NAMESPACE || type == PolarisEntityType.TABLE_LIKE || type == PolarisEntityType.TASK;
    }

    @VisibleForTesting
    public long getEstimatedSize() {
        return this.cache.estimatedSize();
    }

    @VisibleForTesting
    public void invalidateAll() {
        this.cache.invalidateAll();
    }
}

