/*
 *   Licensed to the Apache Software Foundation (ASF) under one
 *   or more contributor license agreements.  See the NOTICE file
 *   distributed with this work for additional information
 *   regarding copyright ownership.  The ASF licenses this file
 *   to you under the Apache License, Version 2.0 (the
 *   "License"); you may not use this file except in compliance
 *   with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *   Unless required by applicable law or agreed to in writing,
 *   software distributed under the License is distributed on an
 *   "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *   KIND, either express or implied.  See the License for the
 *   specific language governing permissions and limitations
 *   under the License.
 *
 */

package org.apache.directory.fortress.web.panel;


import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.directory.fortress.core.util.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.apache.wicket.ajax.AjaxRequestTarget;
import org.apache.wicket.ajax.markup.html.AjaxLink;
import org.apache.wicket.ajax.markup.html.navigation.paging.AjaxPagingNavigator;
import org.apache.directory.fortress.web.modal.ModalWindow;
import org.apache.wicket.markup.html.basic.Label;
import org.apache.wicket.markup.html.list.ListItem;
import org.apache.wicket.markup.html.list.PageableListView;
import org.apache.wicket.markup.html.panel.Panel;
import org.apache.wicket.model.LoadableDetachableModel;
import org.apache.wicket.model.PropertyModel;
import org.apache.wicket.spring.injection.annot.SpringBean;
import org.apache.directory.fortress.web.control.SecUtils;
import org.apache.directory.fortress.core.DelReviewMgr;
import org.apache.directory.fortress.core.model.OrgUnit;


/**
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 * @version $Rev$
 */
public class OUSearchModalPanel extends Panel
{
    /** Default serialVersionUID */
    private static final long serialVersionUID = 1L;
    @SpringBean
    private DelReviewMgr delReviewMgr;
    private static final Logger LOG = LoggerFactory.getLogger( OUSearchModalPanel.class.getName() );
    private ModalWindow window;
    private OrgUnit ouSelection;
    private String ouSearchVal;
    private boolean isUser;


    /**
     * @param id
     */
    public OUSearchModalPanel( String id, ModalWindow window, boolean isUser )
    {
        super( id );
        if (Config.getInstance().getBoolean(org.apache.directory.fortress.core.GlobalIds.IS_ARBAC02))
        {
            this.delReviewMgr.setAdmin(SecUtils.getSession(this));
        }
        this.window = window;
        this.isUser = isUser;
        loadPanel();
    }


    public void loadPanel()
    {
        LoadableDetachableModel requests = getListViewModel();
        PageableListView ouView = createListView( requests );
        add( ouView );
        add( new AjaxPagingNavigator( "navigator", ouView ) );
    }


    private PageableListView createListView( final LoadableDetachableModel requests )
    {
        return new PageableListView( "dataview", requests, 16 )
        {
            /** Default serialVersionUID */
            private static final long serialVersionUID = 1L;


            @Override
            protected void populateItem( final ListItem item )
            {
                final OrgUnit modelObject = ( OrgUnit ) item.getModelObject();
                item.add( new AjaxLink<Void>( "select" )
                {
                    private static final long serialVersionUID = 1L;


                    @Override
                    public void onClick( AjaxRequestTarget target )
                    {
                        ouSelection = modelObject;
                        window.close( target );
                    }
                } );
                item.add( new Label( "name", new PropertyModel( item.getModel(), "name" ) ) );
                item.add( new Label( "description", new PropertyModel( item.getModel(), "description" ) ) );
            }
        };
    }


    private LoadableDetachableModel getListViewModel()
    {
        return new LoadableDetachableModel()
        {
            /** Default serialVersionUID */
            private static final long serialVersionUID = 1L;


            @Override
            protected Object load()
            {
                List<?> ous = null;
                try
                {
                    ouSelection = null;
                    if ( ouSearchVal == null )
                        ouSearchVal = "";
                    if ( isUser )
                        ous = delReviewMgr.search( OrgUnit.Type.USER, ouSearchVal );
                    else
                        ous = delReviewMgr.search( OrgUnit.Type.PERM, ouSearchVal );
                    // sort list by name:
                    if( CollectionUtils.isNotEmpty( ous ))
                    {
                        Collections.sort( ( List<OrgUnit> ) ous, new Comparator<OrgUnit>()
                        {
                            @Override
                            public int compare(OrgUnit o1, OrgUnit o2)
                            {
                                return o1.getName().compareToIgnoreCase( o2.getName() );
                            }
                        } );
                    }

                }
                catch ( org.apache.directory.fortress.core.SecurityException se )
                {
                    String error = "loadPanel caught SecurityException=" + se;
                    LOG.error( error );
                }
                return ous;
            }
        };
    }


    public OrgUnit getSelection()
    {
        return ouSelection;
    }


    public void setSearchVal( String ouSearchVal )
    {
        this.ouSearchVal = ouSearchVal;
    }
}
