/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.maven.cling.invoker;

import java.util.ArrayList;
import java.util.List;

/**
 * CleanArgument
 */
public class CleanArgument {
    public static String[] cleanArgs(String[] args) {
        try {
            return doCleanArgs(args);
        } catch (RuntimeException e) {
            for (String a : args) {
                System.out.println("Arg: '" + a + "'");
            }
            throw e;
        }
    }

    private static String[] doCleanArgs(String[] args) {
        List<String> cleaned = new ArrayList<>();

        StringBuilder currentArg = null;

        for (String arg : args) {
            boolean addedToBuffer = false;

            if (arg.startsWith("\"")) {
                // if we're in the process of building up another arg, push it and start over.
                // this is for the case: "-Dfoo=bar "-Dfoo2=bar two" (note the first unterminated quote)
                if (currentArg != null) {
                    cleaned.add(currentArg.toString());
                }

                // start building an argument here.
                currentArg = new StringBuilder(arg.substring(1));
                addedToBuffer = true;
            }

            // this has to be a separate "if" statement, to capture the case of: "-Dfoo=bar"
            if (addedToBuffer && arg.endsWith("\"")) {
                // if we're building an argument, keep doing so.
                // if this is the case of "-Dfoo=bar", then we need to adjust the buffer.
                if (!currentArg.isEmpty()) {
                    currentArg.setLength(currentArg.length() - 1);
                }

                cleaned.add(currentArg.toString());

                currentArg = null;
                addedToBuffer = false;
                continue;
            }

            // if we haven't added this arg to the buffer, and we ARE building an argument
            // buffer, then append it with a preceding space...again, not sure what else to
            // do other than collapse whitespace.
            // NOTE: The case of a trailing quote is handled by nullifying the arg buffer.
            if (!addedToBuffer) {
                if (currentArg != null) {
                    currentArg.append(' ').append(arg);
                } else {
                    cleaned.add(arg);
                }
            }
        }

        if (currentArg != null) {
            cleaned.add(currentArg.toString());
        }

        int cleanedSz = cleaned.size();

        String[] cleanArgs;

        if (cleanedSz == 0) {
            cleanArgs = args;
        } else {
            cleanArgs = cleaned.toArray(new String[0]);
        }

        return cleanArgs;
    }
}
