/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.compress.archivers.sevenz;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.channels.SeekableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.security.NoSuchAlgorithmException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Random;
import java.util.function.Function;
import javax.crypto.Cipher;
import org.apache.commons.compress.AbstractTest;
import org.apache.commons.compress.MemoryLimitException;
import org.apache.commons.compress.PasswordRequiredException;
import org.apache.commons.compress.archivers.sevenz.SevenZArchiveEntry;
import org.apache.commons.compress.archivers.sevenz.SevenZFile;
import org.apache.commons.compress.archivers.sevenz.SevenZFileOptions;
import org.apache.commons.compress.archivers.sevenz.SevenZMethod;
import org.apache.commons.compress.archivers.sevenz.SevenZMethodConfiguration;
import org.apache.commons.compress.archivers.sevenz.SevenZOutputFile;
import org.apache.commons.compress.utils.MultiReadOnlySeekableByteChannel;
import org.apache.commons.compress.utils.SeekableInMemoryByteChannel;
import org.apache.commons.io.IOUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

class SevenZFileTest
extends AbstractTest {
    private static final String TEST2_CONTENT = "<?xml version = '1.0'?>\r\n<!DOCTYPE connections>\r\n<meinxml>\r\n\t<leer />\r\n</meinxml>\n";

    SevenZFileTest() {
    }

    private static boolean isStrongCryptoAvailable() throws NoSuchAlgorithmException {
        return Cipher.getMaxAllowedKeyLength("AES/ECB/PKCS5Padding") >= 256;
    }

    private void assertDate(SevenZArchiveEntry entry, String value, Function<SevenZArchiveEntry, Boolean> hasValue, Function<SevenZArchiveEntry, FileTime> timeFunction, Function<SevenZArchiveEntry, Date> dateFunction) {
        if (value != null) {
            Assertions.assertTrue((boolean)hasValue.apply(entry));
            Instant parsedInstant = Instant.parse(value);
            FileTime parsedFileTime = FileTime.from(parsedInstant);
            Assertions.assertEquals((Object)parsedFileTime, (Object)timeFunction.apply(entry));
            Assertions.assertEquals((Object)Date.from(parsedInstant), (Object)dateFunction.apply(entry));
        } else {
            Assertions.assertFalse((boolean)hasValue.apply(entry));
            Assertions.assertThrows(UnsupportedOperationException.class, () -> timeFunction.apply(entry));
            Assertions.assertThrows(UnsupportedOperationException.class, () -> dateFunction.apply(entry));
        }
    }

    private void assertDates(SevenZArchiveEntry entry, String modified, String access, String creation) {
        this.assertDate(entry, modified, SevenZArchiveEntry::getHasLastModifiedDate, SevenZArchiveEntry::getLastModifiedTime, SevenZArchiveEntry::getLastModifiedDate);
        this.assertDate(entry, access, SevenZArchiveEntry::getHasAccessDate, SevenZArchiveEntry::getAccessTime, SevenZArchiveEntry::getAccessDate);
        this.assertDate(entry, creation, SevenZArchiveEntry::getHasCreationDate, SevenZArchiveEntry::getCreationTime, SevenZArchiveEntry::getCreationDate);
    }

    private void checkHelloWorld(String fileName) throws Exception {
        try (SevenZFile sevenZFile = this.getSevenZFile(fileName);){
            int bytesRead;
            SevenZArchiveEntry entry = sevenZFile.getNextEntry();
            Assertions.assertEquals((Object)"Hello world.txt", (Object)entry.getName());
            this.assertDates(entry, "2013-05-07T19:40:48Z", null, null);
            byte[] contents = new byte[(int)entry.getSize()];
            for (int off = 0; off < contents.length; off += bytesRead) {
                bytesRead = sevenZFile.read(contents, off, contents.length - off);
                Assertions.assertTrue((bytesRead >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead));
            }
            Assertions.assertEquals((Object)"Hello, world!\n", (Object)new String(contents, StandardCharsets.UTF_8));
            Assertions.assertNull((Object)sevenZFile.getNextEntry());
        }
    }

    private SevenZFile getSevenZFile(String specialPath) throws IOException {
        return ((SevenZFile.Builder)SevenZFile.builder().setFile(SevenZFileTest.getFile(specialPath))).get();
    }

    private byte[] read(SevenZFile sevenZFile, SevenZArchiveEntry entry) throws IOException {
        try (InputStream inputStream = sevenZFile.getInputStream(entry);){
            byte[] byArray = IOUtils.toByteArray((InputStream)inputStream);
            return byArray;
        }
    }

    private byte[] readFully(SevenZFile archive) throws IOException {
        byte[] buf = new byte[1024];
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        int len = 0;
        while ((len = archive.read(buf)) > 0) {
            baos.write(buf, 0, len);
        }
        return baos.toByteArray();
    }

    @Test
    void test7zDecryptUnarchive() throws Exception {
        if (SevenZFileTest.isStrongCryptoAvailable()) {
            this.test7zUnarchive(SevenZFileTest.getFile("bla.encrypted.7z"), SevenZMethod.LZMA, "foo".getBytes(StandardCharsets.UTF_16LE));
        }
    }

    @Test
    void test7zDecryptUnarchiveUsingCharArrayPassword() throws Exception {
        if (SevenZFileTest.isStrongCryptoAvailable()) {
            this.test7zUnarchive(SevenZFileTest.getFile("bla.encrypted.7z"), SevenZMethod.LZMA, "foo".toCharArray());
        }
    }

    @Test
    void test7zDeflate64Unarchive() throws Exception {
        this.test7zUnarchive(SevenZFileTest.getFile("bla.deflate64.7z"), SevenZMethod.DEFLATE64);
    }

    @Test
    void test7zDeflateUnarchive() throws Exception {
        this.test7zUnarchive(SevenZFileTest.getFile("bla.deflate.7z"), SevenZMethod.DEFLATE);
    }

    @Test
    void test7zMultiVolumeUnarchive() throws Exception {
        try (SevenZFile sevenZFile = new SevenZFile(MultiReadOnlySeekableByteChannel.forFiles((File[])new File[]{SevenZFileTest.getFile("bla-multi.7z.001"), SevenZFileTest.getFile("bla-multi.7z.002")}));){
            this.test7zUnarchive(sevenZFile, SevenZMethod.LZMA2);
        }
        sevenZFile = SevenZFile.builder().setSeekableByteChannel(MultiReadOnlySeekableByteChannel.forFiles((File[])new File[]{SevenZFileTest.getFile("bla-multi.7z.001"), SevenZFileTest.getFile("bla-multi.7z.002")})).get();
        try {
            this.test7zUnarchive(sevenZFile, SevenZMethod.LZMA2);
        }
        finally {
            if (sevenZFile != null) {
                sevenZFile.close();
            }
        }
    }

    @Test
    void test7zUnarchive() throws Exception {
        this.test7zUnarchive(SevenZFileTest.getFile("bla.7z"), SevenZMethod.LZMA);
    }

    private void test7zUnarchive(File file, SevenZMethod method) throws Exception {
        this.test7zUnarchive(file, method, false);
    }

    private void test7zUnarchive(File file, SevenZMethod method, boolean tryToRecoverBrokenArchives) throws Exception {
        this.test7zUnarchive(file, method, null, tryToRecoverBrokenArchives);
    }

    private void test7zUnarchive(File file, SevenZMethod method, byte[] password) throws Exception {
        try (SevenZFile sevenZFile = new SevenZFile(file, password);){
            this.test7zUnarchive(sevenZFile, method);
        }
        sevenZFile = ((SevenZFile.Builder)SevenZFile.builder().setFile(file)).setPassword(password).get();
        try {
            this.test7zUnarchive(sevenZFile, method);
        }
        finally {
            if (sevenZFile != null) {
                sevenZFile.close();
            }
        }
    }

    private void test7zUnarchive(File file, SevenZMethod m, char[] password) throws Exception {
        this.test7zUnarchive(file, m, password, false);
    }

    private void test7zUnarchive(File file, SevenZMethod m, char[] password, boolean tryToRecoverBrokenArchives) throws Exception {
        try (SevenZFile sevenZFile = new SevenZFile(file, password, SevenZFileOptions.builder().withTryToRecoverBrokenArchives(tryToRecoverBrokenArchives).build());){
            this.test7zUnarchive(sevenZFile, m);
        }
        sevenZFile = ((SevenZFile.Builder)SevenZFile.builder().setFile(file)).setPassword(password).setTryToRecoverBrokenArchives(tryToRecoverBrokenArchives).get();
        try {
            this.test7zUnarchive(sevenZFile, m);
        }
        finally {
            if (sevenZFile != null) {
                sevenZFile.close();
            }
        }
    }

    private void test7zUnarchive(SevenZFile sevenZFile, SevenZMethod m) throws Exception {
        int bytesRead;
        SevenZArchiveEntry entry = sevenZFile.getNextEntry();
        Assertions.assertEquals((Object)"test1.xml", (Object)entry.getName());
        this.assertDates(entry, "2007-11-14T10:19:02Z", null, null);
        Assertions.assertEquals((Object)m, (Object)((SevenZMethodConfiguration)entry.getContentMethods().iterator().next()).getMethod());
        entry = sevenZFile.getNextEntry();
        Assertions.assertEquals((Object)"test2.xml", (Object)entry.getName());
        this.assertDates(entry, "2007-11-14T10:19:02Z", null, null);
        Assertions.assertEquals((Object)m, (Object)((SevenZMethodConfiguration)entry.getContentMethods().iterator().next()).getMethod());
        byte[] contents = new byte[(int)entry.getSize()];
        for (int off = 0; off < contents.length; off += bytesRead) {
            bytesRead = sevenZFile.read(contents, off, contents.length - off);
            Assertions.assertTrue((bytesRead >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead));
        }
        Assertions.assertEquals((Object)TEST2_CONTENT, (Object)new String(contents, StandardCharsets.UTF_8));
        Assertions.assertNull((Object)sevenZFile.getNextEntry());
    }

    @Test
    void test7zUnarchiveWithDefectHeader() throws Exception {
        this.test7zUnarchive(SevenZFileTest.getFile("bla.noendheaderoffset.7z"), SevenZMethod.LZMA, true);
    }

    @Test
    void test7zUnarchiveWithDefectHeaderFailsByDefault() throws Exception {
        Assertions.assertThrows(IOException.class, () -> this.test7zUnarchive(SevenZFileTest.getFile("bla.noendheaderoffset.7z"), SevenZMethod.LZMA));
    }

    @Test
    void testAllEmptyFilesArchive() throws Exception {
        try (SevenZFile archive = this.getSevenZFile("7z-empty-mhc-off.7z");){
            SevenZArchiveEntry e = archive.getNextEntry();
            Assertions.assertNotNull((Object)e);
            Assertions.assertEquals((Object)"empty", (Object)e.getName());
            this.assertDates(e, "2013-05-14T17:50:19Z", null, null);
            Assertions.assertNull((Object)archive.getNextEntry());
        }
    }

    @Test
    void testCompressedHeaderWithNonDefaultDictionarySize() throws Exception {
        try (SevenZFile sevenZFile = this.getSevenZFile("COMPRESS-256.7z");){
            int count = 0;
            while (sevenZFile.getNextEntry() != null) {
                ++count;
            }
            Assertions.assertEquals((int)446, (int)count);
        }
    }

    @Test
    void testEncryptedArchiveRequiresPassword() throws Exception {
        PasswordRequiredException ex = (PasswordRequiredException)Assertions.assertThrows(PasswordRequiredException.class, () -> this.getSevenZFile("bla.encrypted.7z").close(), (String)"shouldn't decrypt without a password");
        String msg = ex.getMessage();
        Assertions.assertTrue((boolean)msg.startsWith("Cannot read encrypted content from "), (String)"Should start with whining about being unable to decrypt");
        Assertions.assertTrue((boolean)msg.endsWith(" without a password."), (String)"Should finish the sentence properly");
        Assertions.assertTrue((boolean)msg.contains("bla.encrypted.7z"), (String)"Should contain archive's name");
    }

    @Test
    void testExtractNonExistSpecifiedFile() throws Exception {
        try (SevenZFile sevenZFile = this.getSevenZFile("COMPRESS-256.7z");
             SevenZFile anotherSevenZFile = this.getSevenZFile("bla.7z");){
            for (SevenZArchiveEntry nonExistEntry : anotherSevenZFile.getEntries()) {
                Assertions.assertThrows(IllegalArgumentException.class, () -> sevenZFile.getInputStream(nonExistEntry));
            }
        }
    }

    @Test
    void testExtractSpecifiedFile() throws Exception {
        try (SevenZFile sevenZFile = this.getSevenZFile("COMPRESS-256.7z");){
            String testTxtContents = "111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011";
            for (SevenZArchiveEntry entry : sevenZFile.getEntries()) {
                int bytesRead;
                if (!entry.getName().equals("commons-compress-1.7-src/src/test/resources/test.txt")) continue;
                byte[] contents = new byte[(int)entry.getSize()];
                InputStream inputStream = sevenZFile.getInputStream(entry);
                for (int off = 0; off < contents.length; off += bytesRead) {
                    bytesRead = inputStream.read(contents, off, contents.length - off);
                    Assertions.assertTrue((bytesRead >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead));
                }
                Assertions.assertEquals((Object)"111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011", (Object)new String(contents, StandardCharsets.UTF_8));
                break;
            }
        }
    }

    @Test
    void testExtractSpecifiedFileDeprecated() throws Exception {
        try (SevenZFile sevenZFile = new SevenZFile(SevenZFileTest.getFile("COMPRESS-256.7z"));){
            String testTxtContents = "111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011";
            for (SevenZArchiveEntry entry : sevenZFile.getEntries()) {
                int bytesRead;
                if (!entry.getName().equals("commons-compress-1.7-src/src/test/resources/test.txt")) continue;
                byte[] contents = new byte[(int)entry.getSize()];
                InputStream inputStream = sevenZFile.getInputStream(entry);
                for (int off = 0; off < contents.length; off += bytesRead) {
                    bytesRead = inputStream.read(contents, off, contents.length - off);
                    Assertions.assertTrue((bytesRead >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead));
                }
                Assertions.assertEquals((Object)"111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011", (Object)new String(contents, StandardCharsets.UTF_8));
                break;
            }
        }
    }

    @Test
    void testGetDefaultName() throws Exception {
        try (SevenZFile sevenZFile = this.getSevenZFile("bla.deflate64.7z");){
            Assertions.assertEquals((Object)"bla.deflate64", (Object)sevenZFile.getDefaultName());
        }
        sevenZFile = SevenZFile.builder().setSeekableByteChannel(Files.newByteChannel(SevenZFileTest.getFile("bla.deflate64.7z").toPath(), new OpenOption[0])).get();
        try {
            Assertions.assertNull((Object)sevenZFile.getDefaultName());
        }
        finally {
            if (sevenZFile != null) {
                sevenZFile.close();
            }
        }
        sevenZFile = new SevenZFile(Files.newByteChannel(SevenZFileTest.getFile("bla.deflate64.7z").toPath(), new OpenOption[0]), "foo");
        try {
            Assertions.assertEquals((Object)"foo~", (Object)sevenZFile.getDefaultName());
        }
        finally {
            sevenZFile.close();
        }
        sevenZFile = SevenZFile.builder().setSeekableByteChannel(Files.newByteChannel(SevenZFileTest.getFile("bla.deflate64.7z").toPath(), new OpenOption[0])).setDefaultName("foo").get();
        try {
            Assertions.assertEquals((Object)"foo~", (Object)sevenZFile.getDefaultName());
        }
        finally {
            if (sevenZFile != null) {
                sevenZFile.close();
            }
        }
        sevenZFile = SevenZFile.builder().setSeekableByteChannel(Files.newByteChannel(SevenZFileTest.getFile("bla.deflate64.7z").toPath(), new OpenOption[0])).setDefaultName(".foo").get();
        try {
            Assertions.assertEquals((Object)".foo~", (Object)sevenZFile.getDefaultName());
        }
        finally {
            if (sevenZFile != null) {
                sevenZFile.close();
            }
        }
    }

    @Test
    void testGetEntriesOfUnarchiveInMemoryTest() throws IOException {
        byte[] data = SevenZFileTest.readAllBytes("bla.7z");
        try (SevenZFile sevenZFile = SevenZFile.builder().setSeekableByteChannel((SeekableByteChannel)new SeekableInMemoryByteChannel(data)).get();){
            Iterable entries = sevenZFile.getEntries();
            Iterator iter = entries.iterator();
            SevenZArchiveEntry entry = (SevenZArchiveEntry)iter.next();
            Assertions.assertEquals((Object)"test1.xml", (Object)entry.getName());
            entry = (SevenZArchiveEntry)iter.next();
            Assertions.assertEquals((Object)"test2.xml", (Object)entry.getName());
            Assertions.assertFalse((boolean)iter.hasNext());
        }
    }

    @Test
    void testGetEntriesOfUnarchiveTest() throws IOException {
        try (SevenZFile sevenZFile = this.getSevenZFile("bla.7z");){
            Iterable entries = sevenZFile.getEntries();
            Iterator iter = entries.iterator();
            SevenZArchiveEntry entry = (SevenZArchiveEntry)iter.next();
            Assertions.assertEquals((Object)"test1.xml", (Object)entry.getName());
            entry = (SevenZArchiveEntry)iter.next();
            Assertions.assertEquals((Object)"test2.xml", (Object)entry.getName());
            Assertions.assertFalse((boolean)iter.hasNext());
        }
    }

    @Test
    void testGivenNameWinsOverDefaultName() throws Exception {
        SevenZArchiveEntry ae;
        try (SevenZFile sevenZFile = new SevenZFile(SevenZFileTest.getFile("bla.7z"), SevenZFileOptions.builder().withUseDefaultNameForUnnamedEntries(true).build());){
            ae = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)ae);
            Assertions.assertEquals((Object)"test1.xml", (Object)ae.getName());
            ae = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)ae);
            Assertions.assertEquals((Object)"test2.xml", (Object)ae.getName());
            Assertions.assertNull((Object)sevenZFile.getNextEntry());
        }
        sevenZFile = ((SevenZFile.Builder)SevenZFile.builder().setFile(SevenZFileTest.getFile("bla.7z"))).setUseDefaultNameForUnnamedEntries(true).get();
        try {
            ae = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)ae);
            Assertions.assertEquals((Object)"test1.xml", (Object)ae.getName());
            ae = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)ae);
            Assertions.assertEquals((Object)"test2.xml", (Object)ae.getName());
            Assertions.assertNull((Object)sevenZFile.getNextEntry());
        }
        finally {
            if (sevenZFile != null) {
                sevenZFile.close();
            }
        }
    }

    @Test
    void testHandlesEmptyArchiveWithFilesInfo() throws Exception {
        File file = this.newTempFile("empty.7z");
        SevenZOutputFile s = new SevenZOutputFile(file);
        s.close();
        try (SevenZFile z = ((SevenZFile.Builder)SevenZFile.builder().setFile(file)).get();){
            Assertions.assertFalse((boolean)z.getEntries().iterator().hasNext());
            Assertions.assertNull((Object)z.getNextEntry());
        }
    }

    @Test
    void testHandlesEmptyArchiveWithoutFilesInfo() throws Exception {
        try (SevenZFile z = this.getSevenZFile("COMPRESS-492.7z");){
            Assertions.assertFalse((boolean)z.getEntries().iterator().hasNext());
            Assertions.assertNull((Object)z.getNextEntry());
        }
    }

    @Test
    void testHelloWorldHeaderCompressionOffCopy() throws Exception {
        this.checkHelloWorld("7z-hello-mhc-off-copy.7z");
    }

    @Test
    void testHelloWorldHeaderCompressionOffLZMA2() throws Exception {
        this.checkHelloWorld("7z-hello-mhc-off-lzma2.7z");
    }

    @Test
    void testLimitExtractionMemory() {
        Assertions.assertThrows(MemoryLimitException.class, () -> {
            SevenZFile sevenZFile = ((SevenZFile.Builder)SevenZFile.builder().setFile(SevenZFileTest.getFile("bla.7z"))).setMaxMemoryLimitKb(1).get();
            if (sevenZFile != null) {
                sevenZFile.close();
            }
        });
        Assertions.assertThrows(MemoryLimitException.class, () -> {
            SevenZFile sevenZFile = ((SevenZFile.Builder)SevenZFile.builder().setFile(SevenZFileTest.getFile("bla.7z"))).setMaxMemoryLimitKiB(1).get();
            if (sevenZFile != null) {
                sevenZFile.close();
            }
        });
    }

    @Test
    void testNoNameCanBeReplacedByDefaultName() throws Exception {
        try (SevenZFile sevenZFile = ((SevenZFile.Builder)SevenZFile.builder().setFile(SevenZFileTest.getFile("bla-nonames.7z"))).setUseDefaultNameForUnnamedEntries(true).get();){
            SevenZArchiveEntry ae = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)ae);
            Assertions.assertEquals((Object)"bla-nonames", (Object)ae.getName());
            ae = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)ae);
            Assertions.assertEquals((Object)"bla-nonames", (Object)ae.getName());
            Assertions.assertNull((Object)sevenZFile.getNextEntry());
        }
    }

    @Test
    void testNoNameMeansNoNameByDefault() throws Exception {
        try (SevenZFile sevenZFile = this.getSevenZFile("bla-nonames.7z");){
            SevenZArchiveEntry ae = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)ae);
            Assertions.assertNull((Object)ae.getName());
            ae = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)ae);
            Assertions.assertNull((Object)ae.getName());
            Assertions.assertNull((Object)sevenZFile.getNextEntry());
        }
    }

    @Test
    void testNoOOMOnCorruptedHeader() throws IOException {
        ArrayList<Path> testFiles = new ArrayList<Path>();
        testFiles.add(SevenZFileTest.getPath("COMPRESS-542-1.7z"));
        testFiles.add(SevenZFileTest.getPath("COMPRESS-542-2.7z"));
        testFiles.add(SevenZFileTest.getPath("COMPRESS-542-endheadercorrupted.7z"));
        testFiles.add(SevenZFileTest.getPath("COMPRESS-542-endheadercorrupted2.7z"));
        for (Path file : testFiles) {
            IOException e = (IOException)Assertions.assertThrows(IOException.class, () -> {
                SevenZFile sevenZFile = new SevenZFile(Files.newByteChannel(file, new OpenOption[0]), SevenZFileOptions.builder().withTryToRecoverBrokenArchives(true).build());
                sevenZFile.close();
            }, (String)"Expected IOException: start header corrupt and unable to guess end header");
            Assertions.assertEquals((Object)"Start header corrupt and unable to guess end header", (Object)e.getMessage());
            e = (IOException)Assertions.assertThrows(IOException.class, () -> {
                SevenZFile sevenZFile = SevenZFile.builder().setSeekableByteChannel(Files.newByteChannel(file, new OpenOption[0])).setTryToRecoverBrokenArchives(true).get();
                if (sevenZFile != null) {
                    sevenZFile.close();
                }
            }, (String)"Expected IOException: start header corrupt and unable to guess end header");
            Assertions.assertEquals((Object)"Start header corrupt and unable to guess end header", (Object)e.getMessage());
        }
    }

    @Test
    void testRandomAccessMultipleReadSameFile() throws Exception {
        try (SevenZFile sevenZFile = this.getSevenZFile("COMPRESS-256.7z");){
            SevenZArchiveEntry entry;
            String testTxtContents = "111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011";
            SevenZArchiveEntry testTxtEntry = null;
            while ((entry = sevenZFile.getNextEntry()) != null) {
                if (!entry.getName().equals("commons-compress-1.7-src/src/test/resources/test.txt")) continue;
                testTxtEntry = entry;
                break;
            }
            Assertions.assertNotNull(testTxtEntry, (String)"testTxtEntry");
            byte[] contents = new byte[(int)testTxtEntry.getSize()];
            int numberOfReads = 10;
            while (numberOfReads-- > 0) {
                InputStream inputStream = sevenZFile.getInputStream(testTxtEntry);
                try {
                    int bytesRead;
                    for (int off = 0; off < contents.length; off += bytesRead) {
                        bytesRead = inputStream.read(contents, off, contents.length - off);
                        Assertions.assertTrue((bytesRead >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead));
                    }
                    Assertions.assertEquals((Object)SevenZMethod.LZMA2, (Object)((SevenZMethodConfiguration)testTxtEntry.getContentMethods().iterator().next()).getMethod());
                    Assertions.assertEquals((Object)"111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011", (Object)new String(contents, StandardCharsets.UTF_8));
                }
                finally {
                    if (inputStream == null) continue;
                    inputStream.close();
                }
            }
        }
    }

    @Test
    void testRandomAccessTogetherWithSequentialAccess() throws Exception {
        try (SevenZFile sevenZFile = this.getSevenZFile("COMPRESS-256.7z");){
            int bytesRead;
            int off;
            int bytesRead2;
            String testTxtContents = "111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011";
            String filesTxtContents = "0xxxxxxxxx10xxxxxxxx20xxxxxxxx30xxxxxxxx40xxxxxxxx50xxxxxxxx60xxxxxxxx70xxxxxxxx80xxxxxxxx90xxxxxxxx100xxxxxxx110xxxxxxx120xxxxxxx130xxxxxxx -> 0yyyyyyyyy10yyyyyyyy20yyyyyyyy30yyyyyyyy40yyyyyyyy50yyyyyyyy60yyyyyyyy70yyyyyyyy80yyyyyyyy90yyyyyyyy100yyyyyyy110yyyyyyy120yyyyyyy130yyyyyyy\n";
            sevenZFile.getNextEntry();
            SevenZArchiveEntry nextEntry = sevenZFile.getNextEntry();
            byte[] contents = new byte[(int)nextEntry.getSize()];
            Assertions.assertEquals((Object)SevenZMethod.LZMA2, (Object)((SevenZMethodConfiguration)nextEntry.getContentMethods().iterator().next()).getMethod());
            for (off = 0; off < contents.length; off += bytesRead2) {
                bytesRead2 = sevenZFile.read(contents, off, contents.length - off);
                Assertions.assertTrue((bytesRead2 >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead2));
            }
            sevenZFile.getNextEntry();
            sevenZFile.getNextEntry();
            for (SevenZArchiveEntry entry : sevenZFile.getEntries()) {
                int bytesRead3;
                if (!entry.getName().equals("commons-compress-1.7-src/src/test/resources/longsymlink/files.txt")) continue;
                contents = new byte[(int)entry.getSize()];
                InputStream inputStream = sevenZFile.getInputStream(entry);
                for (off = 0; off < contents.length; off += bytesRead3) {
                    bytesRead3 = inputStream.read(contents, off, contents.length - off);
                    Assertions.assertTrue((bytesRead3 >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead3));
                }
                Assertions.assertEquals((Object)SevenZMethod.LZMA2, (Object)((SevenZMethodConfiguration)entry.getContentMethods().iterator().next()).getMethod());
                Assertions.assertEquals((Object)"0xxxxxxxxx10xxxxxxxx20xxxxxxxx30xxxxxxxx40xxxxxxxx50xxxxxxxx60xxxxxxxx70xxxxxxxx80xxxxxxxx90xxxxxxxx100xxxxxxx110xxxxxxx120xxxxxxx130xxxxxxx -> 0yyyyyyyyy10yyyyyyyy20yyyyyyyy30yyyyyyyy40yyyyyyyy50yyyyyyyy60yyyyyyyy70yyyyyyyy80yyyyyyyy90yyyyyyyy100yyyyyyy110yyyyyyy120yyyyyyy130yyyyyyy\n", (Object)new String(contents, StandardCharsets.UTF_8));
                break;
            }
            nextEntry = sevenZFile.getNextEntry();
            while (!nextEntry.getName().equals("commons-compress-1.7-src/src/test/resources/test.txt")) {
                nextEntry = sevenZFile.getNextEntry();
            }
            contents = new byte[(int)nextEntry.getSize()];
            for (off = 0; off < contents.length; off += bytesRead) {
                bytesRead = sevenZFile.read(contents, off, contents.length - off);
                Assertions.assertTrue((bytesRead >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead));
            }
            Assertions.assertEquals((Object)SevenZMethod.LZMA2, (Object)((SevenZMethodConfiguration)nextEntry.getContentMethods().iterator().next()).getMethod());
            Assertions.assertEquals((Object)"111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011", (Object)new String(contents, StandardCharsets.UTF_8));
        }
    }

    @Test
    void testRandomAccessWhenJumpingBackwards() throws Exception {
        try (SevenZFile sevenZFile = this.getSevenZFile("COMPRESS-256.7z");){
            int bytesRead;
            int off;
            int bytesRead2;
            SevenZArchiveEntry entry;
            String testTxtContents = "111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011";
            SevenZArchiveEntry testTxtEntry = null;
            while ((entry = sevenZFile.getNextEntry()) != null) {
                if (!entry.getName().equals("commons-compress-1.7-src/src/test/resources/test.txt")) continue;
                testTxtEntry = entry;
                break;
            }
            SevenZArchiveEntry entryAfterTestTxtEntry = sevenZFile.getNextEntry();
            byte[] entryAfterTestTxtEntryContents = new byte[(int)entryAfterTestTxtEntry.getSize()];
            for (off = 0; off < entryAfterTestTxtEntryContents.length; off += bytesRead2) {
                bytesRead2 = sevenZFile.read(entryAfterTestTxtEntryContents, off, entryAfterTestTxtEntryContents.length - off);
                Assertions.assertTrue((bytesRead2 >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead2));
            }
            Assertions.assertNotNull((Object)testTxtEntry, (String)"testTxtEntry");
            byte[] contents = new byte[(int)testTxtEntry.getSize()];
            try (InputStream inputStream = sevenZFile.getInputStream(testTxtEntry);){
                int bytesRead3;
                for (off = 0; off < contents.length; off += bytesRead3) {
                    bytesRead3 = inputStream.read(contents, off, contents.length - off);
                    Assertions.assertTrue((bytesRead3 >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead3));
                }
                Assertions.assertEquals((Object)SevenZMethod.LZMA2, (Object)((SevenZMethodConfiguration)testTxtEntry.getContentMethods().iterator().next()).getMethod());
                Assertions.assertEquals((Object)"111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011", (Object)new String(contents, StandardCharsets.UTF_8));
            }
            SevenZArchiveEntry nextTestTxtEntry = sevenZFile.getNextEntry();
            byte[] nextTestContents = new byte[(int)nextTestTxtEntry.getSize()];
            for (off = 0; off < nextTestContents.length; off += bytesRead) {
                bytesRead = sevenZFile.read(nextTestContents, off, nextTestContents.length - off);
                Assertions.assertTrue((bytesRead >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead));
            }
            Assertions.assertEquals((Object)nextTestTxtEntry.getName(), (Object)entryAfterTestTxtEntry.getName());
            Assertions.assertEquals((long)nextTestTxtEntry.getSize(), (long)entryAfterTestTxtEntry.getSize());
            Assertions.assertArrayEquals((byte[])nextTestContents, (byte[])entryAfterTestTxtEntryContents);
        }
    }

    @Test
    void testRandomAccessWhenJumpingForwards() throws Exception {
        try (SevenZFile sevenZFile = this.getSevenZFile("COMPRESS-256.7z");){
            SevenZArchiveEntry Entry2;
            String testTxtContents = "111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011";
            SevenZArchiveEntry testTxtEntry = null;
            Iterable entries = sevenZFile.getEntries();
            Iterator iterator = entries.iterator();
            while (iterator.hasNext() && !(testTxtEntry = (Entry2 = (SevenZArchiveEntry)iterator.next())).getName().equals("commons-compress-1.7-src/src/test/resources/test.txt")) {
            }
            SevenZArchiveEntry firstEntry = sevenZFile.getNextEntry();
            byte[] contents = new byte[(int)firstEntry.getSize() / 2];
            sevenZFile.read(contents);
            sevenZFile.getNextEntry();
            SevenZArchiveEntry thirdEntry = sevenZFile.getNextEntry();
            contents = new byte[(int)thirdEntry.getSize() / 2];
            sevenZFile.read(contents);
            Assertions.assertNotNull((Object)testTxtEntry, (String)"testTxtEntry");
            contents = new byte[(int)testTxtEntry.getSize()];
            int numberOfReads = 10;
            while (numberOfReads-- > 0) {
                InputStream inputStream = sevenZFile.getInputStream(testTxtEntry);
                try {
                    int bytesRead;
                    for (int off = 0; off < contents.length; off += bytesRead) {
                        bytesRead = inputStream.read(contents, off, contents.length - off);
                        Assertions.assertTrue((bytesRead >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead));
                    }
                    Assertions.assertEquals((Object)SevenZMethod.LZMA2, (Object)((SevenZMethodConfiguration)testTxtEntry.getContentMethods().iterator().next()).getMethod());
                    Assertions.assertEquals((Object)"111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011\n111111111111111111111111111000101011", (Object)new String(contents, StandardCharsets.UTF_8));
                }
                finally {
                    if (inputStream == null) continue;
                    inputStream.close();
                }
            }
        }
    }

    @Test
    void testRandomlySkippingEntries() throws Exception {
        HashMap<String, byte[]> entriesByName = new HashMap<String, byte[]>();
        try (SevenZFile archive = this.getSevenZFile("COMPRESS-320/Copy.7z");){
            SevenZArchiveEntry entry;
            while ((entry = archive.getNextEntry()) != null) {
                if (!entry.hasStream()) continue;
                entriesByName.put(entry.getName(), this.readFully(archive));
            }
        }
        String[] variants = new String[]{"BZip2-solid.7z", "BZip2.7z", "Copy-solid.7z", "Copy.7z", "Deflate-solid.7z", "Deflate.7z", "LZMA-solid.7z", "LZMA.7z", "LZMA2-solid.7z", "LZMA2.7z"};
        Random rnd = new Random(-559038737L);
        for (String fileName : variants) {
            try (SevenZFile archive = this.getSevenZFile("COMPRESS-320/" + fileName);){
                SevenZArchiveEntry entry;
                while ((entry = archive.getNextEntry()) != null) {
                    if (rnd.nextBoolean() || !entry.hasStream()) continue;
                    Assertions.assertTrue((boolean)entriesByName.containsKey(entry.getName()));
                    byte[] content = this.readFully(archive);
                    Assertions.assertArrayEquals((byte[])content, (byte[])((byte[])entriesByName.get(entry.getName())), (String)("Content mismatch on: " + fileName + "!" + entry.getName()));
                }
            }
        }
    }

    @Test
    void testReadBigSevenZipFile() throws IOException {
        try (SevenZFile sevenZFile = this.getSevenZFile("COMPRESS-592.7z");){
            SevenZArchiveEntry entry = sevenZFile.getNextEntry();
            while (entry != null) {
                if (entry.hasStream()) {
                    byte[] content = new byte[(int)entry.getSize()];
                    sevenZFile.read(content);
                }
                entry = sevenZFile.getNextEntry();
            }
        }
    }

    @Test
    void testReadEntriesOfSize0() throws IOException {
        try (SevenZFile sevenZFile = this.getSevenZFile("COMPRESS-348.7z");){
            int entries = 0;
            SevenZArchiveEntry entry = sevenZFile.getNextEntry();
            while (entry != null) {
                ++entries;
                int b = sevenZFile.read();
                if ("2.txt".equals(entry.getName()) || "5.txt".equals(entry.getName())) {
                    Assertions.assertEquals((int)-1, (int)b);
                } else {
                    Assertions.assertNotEquals((int)-1, (int)b);
                }
                entry = sevenZFile.getNextEntry();
            }
            Assertions.assertEquals((int)5, (int)entries);
        }
    }

    @Test
    void testReadingArchiveProperties() throws IOException {
        String entryName = "COMPRESS-681.txt";
        String entryContent = "https://issues.apache.org/jira/browse/COMPRESS-681";
        try (SevenZFile archive = this.getSevenZFile("COMPRESS-681.7z");){
            int bytesRead;
            SevenZArchiveEntry entry = archive.getNextEntry();
            Assertions.assertEquals((Object)"COMPRESS-681.txt", (Object)entry.getName());
            byte[] contents = new byte[(int)entry.getSize()];
            for (int off = 0; off < contents.length; off += bytesRead) {
                bytesRead = archive.read(contents, off, contents.length - off);
                Assertions.assertTrue((bytesRead >= 0 ? 1 : 0) != 0, (String)("bytesRead exp: >=0, act: " + bytesRead));
            }
            Assertions.assertEquals((Object)"https://issues.apache.org/jira/browse/COMPRESS-681", (Object)new String(contents, StandardCharsets.UTF_8));
            Assertions.assertNull((Object)archive.getNextEntry());
        }
    }

    @Test
    void testReadingBackDeltaDistance() throws Exception {
        SevenZArchiveEntry entry;
        File output = this.newTempFile("delta-distance.7z");
        try (SevenZOutputFile outArchive = new SevenZOutputFile(output);){
            outArchive.setContentMethods(Arrays.asList(new SevenZMethodConfiguration(SevenZMethod.DELTA_FILTER, (Object)32), new SevenZMethodConfiguration(SevenZMethod.LZMA2)));
            entry = new SevenZArchiveEntry();
            entry.setName("foo.txt");
            outArchive.putArchiveEntry(entry);
            outArchive.write(new byte[]{65});
            outArchive.closeArchiveEntry();
        }
        try (SevenZFile archive = ((SevenZFile.Builder)SevenZFile.builder().setFile(output)).get();){
            entry = archive.getNextEntry();
            SevenZMethodConfiguration m = (SevenZMethodConfiguration)entry.getContentMethods().iterator().next();
            Assertions.assertEquals((Object)SevenZMethod.DELTA_FILTER, (Object)m.getMethod());
            Assertions.assertEquals((Object)32, (Object)m.getOptions());
        }
    }

    @Test
    void testReadingBackLZMA2DictSize() throws Exception {
        SevenZArchiveEntry entry;
        File output = this.newTempFile("lzma2-dictsize.7z");
        try (SevenZOutputFile outArchive = new SevenZOutputFile(output);){
            outArchive.setContentMethods(Arrays.asList(new SevenZMethodConfiguration(SevenZMethod.LZMA2, (Object)0x100000)));
            entry = new SevenZArchiveEntry();
            entry.setName("foo.txt");
            outArchive.putArchiveEntry(entry);
            outArchive.write(new byte[]{65});
            outArchive.closeArchiveEntry();
        }
        try (SevenZFile archive = ((SevenZFile.Builder)SevenZFile.builder().setFile(output)).get();){
            entry = archive.getNextEntry();
            SevenZMethodConfiguration m = (SevenZMethodConfiguration)entry.getContentMethods().iterator().next();
            Assertions.assertEquals((Object)SevenZMethod.LZMA2, (Object)m.getMethod());
            Assertions.assertEquals((Object)0x100000, (Object)m.getOptions());
        }
    }

    @Test
    void testReadTimesFromFile() throws IOException {
        try (SevenZFile sevenZFile = this.getSevenZFile("times.7z");){
            SevenZArchiveEntry entry = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)entry);
            Assertions.assertEquals((Object)"test", (Object)entry.getName());
            Assertions.assertTrue((boolean)entry.isDirectory());
            this.assertDates(entry, "2022-03-21T14:50:46.2099751Z", "2022-03-21T14:50:46.2099751Z", "2022-03-16T10:19:24.1051115Z");
            entry = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)entry);
            Assertions.assertEquals((Object)"test/test-times.txt", (Object)entry.getName());
            Assertions.assertFalse((boolean)entry.isDirectory());
            this.assertDates(entry, "2022-03-18T10:00:15Z", "2022-03-18T10:14:37.8130002Z", "2022-03-18T10:14:37.8110032Z");
            entry = sevenZFile.getNextEntry();
            Assertions.assertNotNull((Object)entry);
            Assertions.assertEquals((Object)"test/test-times2.txt", (Object)entry.getName());
            Assertions.assertFalse((boolean)entry.isDirectory());
            this.assertDates(entry, "2022-03-18T10:00:19Z", "2022-03-18T10:14:37.8170038Z", "2022-03-18T10:14:37.8140004Z");
            entry = sevenZFile.getNextEntry();
            Assertions.assertNull((Object)entry);
        }
    }

    @Test
    void testRetrieveInputStreamForAllEntriesMultipleTimes() throws IOException {
        try (SevenZFile sevenZFile = this.getSevenZFile("bla.7z");){
            for (SevenZArchiveEntry entry : sevenZFile.getEntries()) {
                byte[] firstRead = this.read(sevenZFile, entry);
                byte[] secondRead = this.read(sevenZFile, entry);
                Assertions.assertArrayEquals((byte[])firstRead, (byte[])secondRead);
            }
        }
    }

    @Test
    void testRetrieveInputStreamForAllEntriesWithoutCRCMultipleTimes() throws IOException {
        try (SevenZOutputFile out = new SevenZOutputFile(this.newTempFile("test.7z"));){
            Path inputFile = Files.createTempFile("SevenZTestTemp", "", new FileAttribute[0]);
            SevenZArchiveEntry entry = out.createArchiveEntry(inputFile.toFile(), "test.txt");
            out.putArchiveEntry(entry);
            out.write("Test".getBytes(StandardCharsets.UTF_8));
            out.closeArchiveEntry();
            Files.deleteIfExists(inputFile);
        }
        try (SevenZFile sevenZFile = ((SevenZFile.Builder)SevenZFile.builder().setFile(this.newTempFile("test.7z"))).get();){
            for (SevenZArchiveEntry entry : sevenZFile.getEntries()) {
                byte[] firstRead = this.read(sevenZFile, entry);
                byte[] secondRead = this.read(sevenZFile, entry);
                Assertions.assertArrayEquals((byte[])firstRead, (byte[])secondRead);
            }
        }
    }

    @Test
    void testRetrieveInputStreamForShuffledEntries() throws IOException {
        try (SevenZFile sevenZFile = this.getSevenZFile("COMPRESS-348.7z");){
            List entries = (List)sevenZFile.getEntries();
            Collections.shuffle(entries);
            for (SevenZArchiveEntry entry : entries) {
                this.read(sevenZFile, entry);
            }
        }
    }

    @Test
    void testSevenZWithEOS() throws IOException {
        try (SevenZFile sevenZFile = this.getSevenZFile("lzma-with-eos.7z");){
            List entries = (List)sevenZFile.getEntries();
            for (SevenZArchiveEntry entry : entries) {
                this.read(sevenZFile, entry);
            }
        }
    }

    @Test
    void testSignatureCheck() {
        Assertions.assertTrue((boolean)SevenZFile.matches((byte[])SevenZFile.sevenZSignature, (int)SevenZFile.sevenZSignature.length));
        Assertions.assertTrue((boolean)SevenZFile.matches((byte[])SevenZFile.sevenZSignature, (int)(SevenZFile.sevenZSignature.length + 1)));
        Assertions.assertFalse((boolean)SevenZFile.matches((byte[])SevenZFile.sevenZSignature, (int)(SevenZFile.sevenZSignature.length - 1)));
        Assertions.assertFalse((boolean)SevenZFile.matches((byte[])new byte[]{1, 2, 3, 4, 5, 6}, (int)6));
        Assertions.assertTrue((boolean)SevenZFile.matches((byte[])new byte[]{55, 122, -68, -81, 39, 28}, (int)6));
        Assertions.assertFalse((boolean)SevenZFile.matches((byte[])new byte[]{55, 122, -68, -81, 39, 29}, (int)6));
    }
}

